// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- > pi+pi+ omega or eta
  class CMD2_2000_I532970 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CMD2_2000_I532970);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      book(_numOmegaPiPi, 1, 1, 1);
      book(_numEtaPiPi  , 2, 1, 1);
      for (const string& en : _numOmegaPiPi.binning().edges<0>()) {
        const double eval = stod(en)*GeV;
        if (isCompatibleWithSqrtS(eval)) {
          _sqs = en; break;
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      const FinalState& fs = apply<FinalState>(event, "FS");

      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles()) {
        if (p.children().empty()) continue;
        // find the eta
        if (p.pid()==221) {
          map<long,int> nRes = nCount;
          int ncount = ntotal;
          findChildren(p,nRes,ncount);
          // eta pi+pi-
          if (ncount!=2) continue;
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==211) {
              if (val.second !=1) {
                matched = false;
                break;
              }
            }
            else if(val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            _numEtaPiPi->fill(_sqs);
          }
        }
        else if(p.pid()==223) {
          // find the omega
          map<long,int> nRes = nCount;
          int ncount = ntotal;
          findChildren(p,nRes,ncount);
          // eta pi+pi-
          if (ncount!=2) continue;
          bool matched = true;
          for (const auto& val : nRes) {
            if (abs(val.first)==211) {
              if (val.second !=1) {
                matched = false;
                break;
              }
            }
            else if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            _numOmegaPiPi->fill(_sqs);
          }
        }
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      double fact = crossSection()/sumOfWeights()/nanobarn;
      scale(_numOmegaPiPi,fact);
      scale(_numEtaPiPi,  fact);
    }

    /// @}

    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _numEtaPiPi,_numOmegaPiPi;
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(CMD2_2000_I532970);
}
