import yoda

# Numerical data required to derived forward neutrino flux taken from arXiv:2105.08270.
# Neutrino cross-sections calculated with GENIE from arXiv:2402.13318.

# Neutrino PDG IDs
neutrinos = [ 12, -12, 14, -14, 16, -16 ]

# Hadron PDG IDs
hadrons = [ 211, -211, 321, -321, 310, 130, 3122, -3122, 3222,
           -3222, 3112, -3112, 3322, -3322, 3312, -3312,
           411, -411, 421, -421, 431, -431, 4122, -4122 ]

# Neutrino energy binning
energies = yoda.Axis([ 1.0000e+01, 1.2589e+01, 1.5849e+01, 1.9953e+01, 2.5119e+01,
             3.1623e+01, 3.9811e+01, 5.0119e+01, 6.3096e+01, 7.9433e+01,
             1.0000e+02, 1.2589e+02, 1.5849e+02, 1.9953e+02, 2.5119e+02,
             3.1623e+02, 3.9811e+02, 5.0119e+02, 6.3096e+02, 7.9433e+02,
             1.0000e+03, 1.2589e+03, 1.5849e+03, 1.9953e+03, 2.5119e+03,
             3.1623e+03, 3.9811e+03, 5.0119e+03, 6.3096e+03, 7.9433e+03,
             1.0000e+04 ])

# Neutrino cross-section values
xsint = {
   12 : [ 8.0014e-39, 7.8553e-39, 7.7334e-39, 7.6364e-39, 7.5567e-39, 7.4907e-39, 7.4354e-39,
          7.3869e-39, 7.3426e-39, 7.3009e-39, 7.2604e-39, 7.2199e-39, 7.1771e-39, 7.1331e-39,
          7.0848e-39, 7.0325e-39, 6.9748e-39, 6.9099e-39, 6.8382e-39, 6.7557e-39, 6.6633e-39,
          6.5581e-39, 6.4387e-39, 6.3050e-39, 6.1530e-39, 5.9848e-39, 5.7936e-39, 5.5893e-39,
          5.3729e-39, 5.1468e-39, 4.9250e-39 ],
  -12 : [ 3.6445e-39, 3.5323e-39, 3.4387e-39, 3.3616e-39, 3.3009e-39, 3.2555e-39, 3.2211e-39,
          3.1976e-39, 3.1827e-39, 3.1745e-39, 3.1706e-39, 3.1702e-39, 3.1720e-39, 3.1756e-39,
          3.1795e-39, 3.1832e-39, 3.1862e-39, 3.1878e-39, 3.1876e-39, 3.1853e-39, 3.1802e-39,
          3.1719e-39, 3.1601e-39, 3.1444e-39, 3.1243e-39, 3.0995e-39, 3.0656e-39, 3.0304e-39,
          2.9928e-39, 2.9517e-39, 2.9100e-39 ],
   14 : [ 7.9486e-39, 7.8132e-39, 7.6944e-39, 7.6067e-39, 7.5380e-39, 7.4725e-39, 7.4207e-39,
          7.3753e-39, 7.3335e-39, 7.2948e-39, 7.2557e-39, 7.2171e-39, 7.1759e-39, 7.1331e-39,
          7.0858e-39, 7.0343e-39, 6.9773e-39, 6.9128e-39, 6.8414e-39, 6.7592e-39, 6.6672e-39,
          6.5623e-39, 6.4430e-39, 6.3096e-39, 6.1574e-39, 5.9897e-39, 5.7994e-39, 5.5947e-39,
          5.3781e-39, 5.1520e-39, 4.9306e-39 ],
  -14 : [ 3.6166e-39, 3.5124e-39, 3.4214e-39, 3.3474e-39, 3.2897e-39, 3.2441e-39, 3.2121e-39,
          3.1895e-39, 3.1750e-39, 3.1674e-39, 3.1645e-39, 3.1642e-39, 3.1666e-39, 3.1705e-39,
          3.1748e-39, 3.1790e-39, 3.1823e-39, 3.1840e-39, 3.1841e-39, 3.1818e-39, 3.1769e-39,
          3.1688e-39, 3.1571e-39, 3.1415e-39, 3.1213e-39, 3.0968e-39, 3.0639e-39, 3.0279e-39,
          2.9902e-39, 2.9491e-39, 2.9075e-39 ],
   16 : [ 2.0234e-39, 2.4690e-39, 2.9149e-39, 3.3532e-39, 3.7734e-39, 4.1726e-39, 4.5356e-39,
          4.8674e-39, 5.1610e-39, 5.4183e-39, 5.6404e-39, 5.8270e-39, 5.9831e-39, 6.1097e-39,
          6.2096e-39, 6.2836e-39, 6.3344e-39, 6.3623e-39, 6.3690e-39, 6.3541e-39, 6.3183e-39,
          6.2613e-39, 6.1824e-39, 6.0832e-39, 5.9602e-39, 5.8170e-39, 5.6495e-39, 5.4649e-39,
          5.2643e-39, 5.0527e-39, 4.8442e-39 ],
  -16 : [ 8.8459e-40, 1.1039e-39, 1.3059e-39, 1.4873e-39, 1.6511e-39, 1.8015e-39, 1.9381e-39,
          2.0657e-39, 2.1830e-39, 2.2921e-39, 2.3922e-39, 2.4809e-39, 2.5639e-39, 2.6400e-39,
          2.7093e-39, 2.7714e-39, 2.8263e-39, 2.8740e-39, 2.9146e-39, 2.9479e-39, 2.9744e-39,
          2.9934e-39, 3.0055e-39, 3.0107e-39, 3.0085e-39, 2.9995e-39, 2.9817e-39, 2.9590e-39,
          2.9311e-39, 2.8986e-39, 2.8641e-39 ],
}


# Hadron decay branchings
decay_branchings = {
   211  : [ 0.9998770, 0 ],
  -211  : [ 0, 0.9998770 ],
   321  : [ 0.6690563, 0 ],
  -321  : [ 0, 0.6690563 ],
   310  : [ 0.0002345, 0.0002345 ],
   130  : [ 0.1352191, 0.1352191 ],
   3122 : [ 0, 0.0001570 ],
  -3122 : [ 0.0001570, 0 ],
   3222 : [ 0, 0 ],
  -3222 : [ 0, 0 ],
   3112 : [ 0.0004500, 0 ],
  -3112 : [ 0, 0.0004500 ],
   3322 : [ 0.0000046, 0 ],
  -3322 : [ 0, 0.0000046 ],
   3312 : [ 0, 0.0003500 ],
  -3312 : [ 0.0003500, 0 ],
   411  : [ 0.16611, -1 ],
  -411  : [ -1, 0.16611 ],
   421  : [ 0.06566, -1 ],
  -421  : [ -1, 0.06566 ],
   431  : [ 0.08688, -1 ],
  -431  : [ -1, 0.08688 ],
   4122 : [ 0.04485, -1 ],
  -4122 : [ -1, 0.04485 ],
}

# CDF energy values for hadrons
cdf_energies = {
   130 : [[6.80000000e-04, 3.03000000e-03, 7.87000000e-03, 1.65300000e-02,
           3.06400000e-02, 4.95900000e-02, 7.34000000e-02, 1.02610000e-01,
           1.37290000e-01, 1.78820000e-01, 2.25100000e-01, 2.77190000e-01,
           3.34050000e-01, 3.95020000e-01, 4.59540000e-01, 5.26660000e-01,
           5.94170000e-01, 6.62400000e-01, 7.28510000e-01, 7.91530000e-01,
           8.50980000e-01, 9.04530000e-01, 9.47870000e-01, 9.82130000e-01,
           1.00000000e+00],
          [6.74672529e-03, 1.41580181e-02, 2.15693109e-02, 2.89806037e-02,
           3.63918965e-02, 4.38031894e-02, 5.12144822e-02, 5.86257750e-02,
           6.60370678e-02, 7.34483606e-02, 8.08596534e-02, 8.82709462e-02,
           9.56822391e-02, 1.03093532e-01, 1.10504825e-01, 1.17916118e-01,
           1.25327410e-01, 1.32738703e-01, 1.40149996e-01, 1.47561289e-01,
           1.54972582e-01, 1.62383874e-01, 1.69795167e-01, 1.77206460e-01,
           1.84617753e-01]],
   310 : [[6.80000000e-04, 3.03000000e-03, 7.87000000e-03, 1.65300000e-02,
           3.06400000e-02, 4.95900000e-02, 7.34000000e-02, 1.02610000e-01,
           1.37290000e-01, 1.78820000e-01, 2.25100000e-01, 2.77190000e-01,
           3.34050000e-01, 3.95020000e-01, 4.59540000e-01, 5.26660000e-01,
           5.94170000e-01, 6.62400000e-01, 7.28510000e-01, 7.91530000e-01,
           8.50980000e-01, 9.04530000e-01, 9.47870000e-01, 9.82130000e-01,
           1.00000000e+00],
          [6.74672529e-03, 1.41580181e-02, 2.15693109e-02, 2.89806037e-02,
           3.63918965e-02, 4.38031894e-02, 5.12144822e-02, 5.86257750e-02,
           6.60370678e-02, 7.34483606e-02, 8.08596534e-02, 8.82709462e-02,
           9.56822391e-02, 1.03093532e-01, 1.10504825e-01, 1.17916118e-01,
           1.25327410e-01, 1.32738703e-01, 1.40149996e-01, 1.47561289e-01,
           1.54972582e-01, 1.62383874e-01, 1.69795167e-01, 1.77206460e-01,
           1.84617753e-01]],
   211 : [[0., 1.], [0.02968, 0.02984]],
  3112 : [[4.30000000e-04, 2.45000000e-03, 6.94000000e-03, 1.52600000e-02,
           2.78400000e-02, 4.55100000e-02, 6.81900000e-02, 9.71400000e-02,
           1.32770000e-01, 1.73630000e-01, 2.19550000e-01, 2.69950000e-01,
           3.25390000e-01, 3.84270000e-01, 4.47400000e-01, 5.12530000e-01,
           5.80150000e-01, 6.45650000e-01, 7.12970000e-01, 7.76420000e-01,
           8.36140000e-01, 8.92760000e-01, 9.39830000e-01, 9.77930000e-01,
           1.00000000e+00],
          [4.06127508e-03, 9.71210294e-03, 1.53629308e-02, 2.10137587e-02,
           2.66645865e-02, 3.23154144e-02, 3.79662422e-02, 4.36170701e-02,
           4.92678980e-02, 5.49187258e-02, 6.05695537e-02, 6.62203815e-02,
           7.18712094e-02, 7.75220373e-02, 8.31728651e-02, 8.88236930e-02,
           9.44745208e-02, 1.00125349e-01, 1.05776177e-01, 1.11427004e-01,
           1.17077832e-01, 1.22728660e-01, 1.28379488e-01, 1.34030316e-01,
           1.39681144e-01]],
  3122 : [[5.00000000e-04, 2.23000000e-03, 6.36000000e-03, 1.36100000e-02,
           2.40300000e-02, 3.91500000e-02, 5.90300000e-02, 8.46700000e-02,
           1.14650000e-01, 1.51720000e-01, 1.93350000e-01, 2.39750000e-01,
           2.91240000e-01, 3.47410000e-01, 4.07390000e-01, 4.71650000e-01,
           5.38090000e-01, 6.05310000e-01, 6.75930000e-01, 7.43480000e-01,
           8.09920000e-01, 8.71780000e-01, 9.27530000e-01, 9.72790000e-01,
           1.00000000e+00],
          [2.30650351e-03, 5.04623569e-03, 7.78596787e-03, 1.05257001e-02,
           1.32654322e-02, 1.60051644e-02, 1.87448966e-02, 2.14846288e-02,
           2.42243610e-02, 2.69640931e-02, 2.97038253e-02, 3.24435575e-02,
           3.51832897e-02, 3.79230219e-02, 4.06627541e-02, 4.34024862e-02,
           4.61422184e-02, 4.88819506e-02, 5.16216828e-02, 5.43614150e-02,
           5.71011471e-02, 5.98408793e-02, 6.25806115e-02, 6.53203437e-02,
           6.80600759e-02]],
   321 : [[1.00000000e-04, 3.50000000e-04, 8.90000000e-04, 1.68000000e-03,
           2.77000000e-03, 4.50000000e-03, 6.65000000e-03, 9.38000000e-03,
           1.25800000e-02, 1.59600000e-02, 1.95300000e-02, 2.36000000e-02,
           2.78600000e-02, 3.22300000e-02, 3.64500000e-02, 4.00400000e-02,
           4.36400000e-02, 4.67100000e-02, 4.88100000e-02, 4.96700000e-02,
           1.00000000e+00],
          [9.80326769e-03, 1.90167280e-02, 2.82301882e-02, 3.74436485e-02,
           4.66571088e-02, 5.58705691e-02, 6.50840293e-02, 7.42974896e-02,
           8.35109499e-02, 9.27244102e-02, 1.01937870e-01, 1.11151331e-01,
           1.20364791e-01, 1.29578251e-01, 1.38791712e-01, 1.48005172e-01,
           1.57218632e-01, 1.66432092e-01, 1.75645553e-01, 1.84859013e-01,
           2.30926314e-01]],
  3312 : [[4.00000000e-04, 2.39000000e-03, 7.07000000e-03, 1.47500000e-02,
           2.70400000e-02, 4.31500000e-02, 6.37700000e-02, 9.01000000e-02,
           1.21390000e-01, 1.58970000e-01, 2.02580000e-01, 2.50730000e-01,
           3.04170000e-01, 3.61950000e-01, 4.23540000e-01, 4.87340000e-01,
           5.53560000e-01, 6.20480000e-01, 6.87520000e-01, 7.54580000e-01,
           8.19520000e-01, 8.79420000e-01, 9.32630000e-01, 9.75670000e-01,
           1.00000000e+00],
          [2.91161719e-03, 6.73347081e-03, 1.05553244e-02, 1.43771781e-02,
           1.81990317e-02, 2.20208853e-02, 2.58427390e-02, 2.96645926e-02,
           3.34864462e-02, 3.73082998e-02, 4.11301535e-02, 4.49520071e-02,
           4.87738607e-02, 5.25957143e-02, 5.64175680e-02, 6.02394216e-02,
           6.40612752e-02, 6.78831289e-02, 7.17049825e-02, 7.55268361e-02,
           7.93486897e-02, 8.31705434e-02, 8.69923970e-02, 9.08142506e-02,
           9.46361043e-02]],
  3322 : [[3.60000000e-04, 2.09000000e-03, 5.81000000e-03, 1.22300000e-02,
           2.15800000e-02, 3.45900000e-02, 5.27500000e-02, 7.49400000e-02,
           1.02110000e-01, 1.34120000e-01, 1.71640000e-01, 2.14100000e-01,
           2.62180000e-01, 3.15690000e-01, 3.75100000e-01, 4.37400000e-01,
           5.03210000e-01, 5.71270000e-01, 6.43770000e-01, 7.15450000e-01,
           7.85210000e-01, 8.54730000e-01, 9.17510000e-01, 9.68400000e-01,
           1.00000000e+00],
          [6.95435752e-04, 1.47018743e-03, 2.24493911e-03, 3.01969080e-03,
           3.79444248e-03, 4.56919416e-03, 5.34394584e-03, 6.11869752e-03,
           6.89344920e-03, 7.66820088e-03, 8.44295256e-03, 9.21770424e-03,
           9.99245593e-03, 1.07672076e-02, 1.15419593e-02, 1.23167110e-02,
           1.30914627e-02, 1.38662143e-02, 1.46409660e-02, 1.54157177e-02,
           1.61904694e-02, 1.69652211e-02, 1.77399727e-02, 1.85147244e-02,
           1.92894761e-02]],
   411 : [[0.00143, 0.0066 , 0.01779, 0.0359 , 0.06308,
           0.09736, 0.14047, 0.19267, 0.25056, 0.31516,
           0.38533, 0.4603 , 0.53944, 0.61904, 0.69671,
           0.76859, 0.82864, 0.88298, 0.93119, 0.9717 ,
           0.99315, 0.9955 , 0.99756, 1.],
          [0.02, 0.06, 0.1, 0.14, 0.18,
           0.22, 0.26, 0.3, 0.34, 0.38,
           0.42, 0.46, 0.5, 0.54, 0.58,
           0.62, 0.66, 0.7, 0.74, 0.78,
           0.82, 0.86, 0.9, 0.94 ]],
  4122 : [[5.3000e-04, 4.2600e-03, 1.3230e-02, 2.9650e-02, 5.5470e-02,
           9.0560e-02, 1.3608e-01, 1.9190e-01, 2.5644e-01, 3.3082e-01,
           4.1116e-01, 4.9705e-01, 5.8394e-01, 6.6998e-01, 7.5044e-01,
           8.2409e-01, 8.8539e-01, 9.3578e-01, 9.7089e-01, 9.9116e-01,
           9.9686e-01, 9.9939e-01, 1.0000e+00, 1.0000e+00, 1.0000e+00],
          [2.0000e-02, 6.0000e-02, 1.0000e-01, 1.4000e-01, 1.8000e-01,
           2.2000e-01, 2.6000e-01, 3.0000e-01, 3.4000e-01, 3.8000e-01,
           4.2000e-01, 4.6000e-01, 5.0000e-01, 5.4000e-01, 5.8000e-01,
           6.2000e-01, 6.6000e-01, 7.0000e-01, 7.4000e-01, 7.8000e-01,
           8.2000e-01, 8.6000e-01, 9.0000e-01, 9.4000e-01, 9.8000e-01]],
   421 : [[0.00113, 0.00621, 0.01742, 0.03612, 0.0625 ,
           0.09749, 0.14195,  0.192 , 0.25073, 0.31636,
           0.38674, 0.46105, 0.53936, 0.61836, 0.69505,
           0.76506, 0.82535, 0.87729, 0.92687, 0.96856,
           0.99192, 0.99651, 1.],
          [0.02, 0.06, 0.1 , 0.14, 0.18,
           0.22, 0.26, 0.3 , 0.34, 0.38,
           0.42, 0.46, 0.5 , 0.54, 0.58,
           0.62, 0.66, 0.7 , 0.74, 0.78,
           0.82, 0.86, 0.9]],
   431 : [[5.6000e-04, 4.1300e-03, 1.2960e-02, 2.8820e-02, 5.5100e-02,
           9.1320e-02, 1.3923e-01, 1.9547e-01, 2.6281e-01, 3.3909e-01,
           4.2140e-01, 5.0623e-01, 5.9237e-01, 6.7092e-01, 7.3848e-01,
           7.8896e-01, 8.3460e-01, 8.7674e-01, 9.0920e-01, 9.2550e-01,
           9.2676e-01, 9.2693e-01, 1.0000e+00, 1.0000e+00, 1.0000e+00],
          [2.2000e-02, 6.6000e-02, 1.1000e-01, 1.5400e-01, 1.9800e-01,
           2.4200e-01, 2.8600e-01, 3.3000e-01, 3.7400e-01, 4.1800e-01,
           4.6200e-01, 5.0600e-01, 5.5000e-01, 5.9400e-01, 6.3800e-01,
           6.8200e-01, 7.2600e-01, 7.7000e-01, 8.1400e-01, 8.5800e-01,
           9.0200e-01, 9.4600e-01, 9.9000e-01, 1.0340e+00, 1.0780e+00]],
}

aper_edges = yoda.Axis([ -1.05000e+01,   1.05000000e+01, 1.06000000e+01, 1.44000000e+01,
                         1.45000000e+01, 1.87000000e+01, 1.88000000e+01, 1.89000000e+01,
                         1.90000000e+01, 1.91000000e+01, 2.09000000e+01, 2.10000000e+01,
                         2.11000000e+01, 2.12000000e+01, 2.13000000e+01, 2.15000000e+01,
                         2.17000000e+01, 2.18000000e+01, 2.19000000e+01, 2.20000000e+01,
                         2.21000000e+01, 2.22000000e+01, 2.23000000e+01, 2.24000000e+01,
                         2.25000000e+01, 2.26000000e+01, 3.13000000e+01, 3.14000000e+01,
                         3.17000000e+01, 4.51000000e+01, 4.52000000e+01, 4.53000000e+01,
                         4.54000000e+01, 4.55000000e+01, 4.56000000e+01, 4.57000000e+01,
                         4.58000000e+01, 4.59000000e+01, 4.60000000e+01, 4.61000000e+01,
                         4.62000000e+01, 4.63000000e+01, 4.64000000e+01, 4.65000000e+01,
                         5.48000000e+01, 5.49000000e+01, 5.50000000e+01, 5.77000000e+01,
                         5.78000000e+01, 5.79000000e+01, 5.80000000e+01, 5.81000000e+01,
                         5.82000000e+01, 5.85000000e+01, 5.86000000e+01, 5.87000000e+01,
                         5.88000000e+01, 5.89000000e+01, 5.90000000e+01, 5.91000000e+01,
                         5.92000000e+01, 5.93000000e+01, 5.94000000e+01, 5.95000000e+01,
                         5.96000000e+01, 5.97000000e+01, 8.44000000e+01, 8.45000000e+01,
                         8.46000000e+01, 8.47000000e+01, 1.39300000e+02, 1.41200000e+02,
                         1.44800000e+02, 1.44900000e+02, 1.45000000e+02, 1.49530000e+02,
                         1.50530000e+02, 1.50600000e+02, 1.50700000e+02, 1.50800000e+02,
                         1.50900000e+02, 1.51000000e+02, 1.51100000e+02, 1.51300000e+02,
                         1.51400000e+02, 1.51500000e+02, 1.51600000e+02, 1.51700000e+02,
                         1.51800000e+02, 1.51900000e+02, 1.52000000e+02, 1.52100000e+02,
                         1.52200000e+02, 1.52300000e+02, 1.52400000e+02, 1.52500000e+02,
                         1.52600000e+02, 1.63299999e+02, 1.63399999e+02, 1.63499999e+02,
                         1.72199999e+02, 1.72399999e+02, 1.72499999e+02, 1.83856999e+02,
                         1.84856999e+02, 1.92399999e+02, 1.92499999e+02, 1.99499999e+02,
                         1.99599999e+02, 1.99999999e+02, 2.00099999e+02, 2.18512999e+02,
                         2.19512999e+02, 2.24299999e+02, 2.24399999e+02, 2.31399999e+02,
                         2.31499999e+02, 2.31899999e+02, 2.31999999e+02, 2.56499999e+02,
                         2.56599999e+02, 2.58799999e+02, 2.58899999e+02, 2.58999999e+02,
                         2.59399999e+02, 2.59499999e+02 ])

aper_vals = [[ 2.90000000e-02, 3.81667000e-02, 4.00000000e-02, 5.56757000e-02,
               6.00000000e-02, 5.46248000e-02, 4.38745000e-02, 3.31242000e-02,
               2.23750000e-02, 1.70000000e-02, 1.87858000e-02, 2.23571000e-02,
               2.59286000e-02, 2.95000000e-02, 3.00000000e-02, 2.98044000e-02,
               2.77455000e-02, 2.67160000e-02, 2.56866000e-02, 2.46572000e-02,
               2.36278000e-02, 2.25983000e-02, 2.15689000e-02, 2.05395000e-02,
               1.95100000e-02, 1.89500000e-02, 2.42409000e-02, 3.01000000e-02,
               2.40000000e-02, 2.53845000e-02, 2.78703000e-02, 3.03560000e-02,
               3.10282000e-02, 3.03873000e-02, 2.97463000e-02, 2.91053000e-02,
               2.84643000e-02, 2.78233000e-02, 2.71824000e-02, 2.65414000e-02,
               2.59005000e-02, 2.52595000e-02, 2.46185000e-02, 2.40000000e-02,
               2.49518000e-02, 2.75241000e-02, 2.88000000e-02, 3.00800000e-02,
               3.21817000e-02, 3.42835000e-02, 3.63851000e-02, 3.84868000e-02,
               4.00000000e-02, 4.08859000e-02, 4.29460000e-02, 4.50061000e-02,
               4.70662000e-02, 4.91263000e-02, 5.11863000e-02, 5.32464000e-02,
               5.53064000e-02, 5.73665000e-02, 5.94266000e-02, 6.14867000e-02,
               6.35468000e-02, 6.40000000e-02, 6.97931000e-02, 8.18621000e-02,
               9.39310000e-02, 1.06000000e-01, 4.00000000e-02, 2.60000000e-02,
               3.06668000e-02, 3.53336000e-02, 4.00000000e-02, 1.13210117e-02,
               3.66992000e-02, 3.54767000e-02, 3.42542000e-02, 3.30317000e-02,
               3.18092000e-02, 3.05868000e-02, 3.00000000e-02, 3.00841000e-02,
               3.01854000e-02, 3.02867000e-02, 3.03880000e-02, 3.04894000e-02,
               3.05907000e-02, 3.06920000e-02, 3.07933000e-02, 3.08946000e-02,
               3.09959000e-02, 3.10973000e-02, 3.11986000e-02, 3.12999000e-02,
               3.13000000e-02, 3.01587000e-02, 2.89599000e-02, 2.89000000e-02,
               3.00000000e-02, 3.50000000e-02, 4.00000000e-02, 6.06333266e-03,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 2.06973146e-03,
               4.00000000e-02, 2.07120000e-02, 1.76500000e-02, 2.17931000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 3.25519000e-02,
               2.89000000e-02, 2.72182000e-02, 2.46229000e-02, 2.40000000e-02,
               2.29290000e-02, 2.20000000e-02 ],
             [ 2.90000000e-02, 3.81667000e-02, 4.00000000e-02, 5.56757000e-02,
               6.00000000e-02, 5.46248000e-02, 4.38745000e-02, 3.31242000e-02,
               2.23750000e-02, 1.70000000e-02, 1.87858000e-02, 2.23571000e-02,
               2.59286000e-02, 2.95000000e-02, 3.00000000e-02, 2.98911000e-02,
               2.87452000e-02, 2.81723000e-02, 2.75993000e-02, 2.70264000e-02,
               2.64535000e-02, 2.58805000e-02, 2.53076000e-02, 2.47346000e-02,
               2.41617000e-02, 2.38500000e-02, 2.68158000e-02, 3.01000000e-02,
               2.89000000e-02, 2.93678000e-02, 3.02075000e-02, 3.10473000e-02,
               3.12744000e-02, 3.10579000e-02, 3.08413000e-02, 3.06248000e-02,
               3.04082000e-02, 3.01917000e-02, 2.99751000e-02, 2.97586000e-02,
               2.95421000e-02, 2.93255000e-02, 2.91090000e-02, 2.89000000e-02,
               2.98518000e-02, 3.24241000e-02, 3.37000000e-02, 3.44200000e-02,
               3.56022000e-02, 3.67845000e-02, 3.79666000e-02, 3.91488000e-02,
               4.00000000e-02, 3.95017000e-02, 3.83429000e-02, 3.71841000e-02,
               3.60252000e-02, 3.48664000e-02, 3.37077000e-02, 3.25489000e-02,
               3.13901000e-02, 3.02313000e-02, 2.90725000e-02, 2.79137000e-02,
               2.67549000e-02, 2.65000000e-02, 3.74655000e-02, 6.03103000e-02,
               8.31552000e-02, 1.06000000e-01, 4.00000000e-02, 2.60000000e-02,
               3.06668000e-02, 3.53336000e-02, 4.00000000e-02, 5.00000000e-02,
               3.66992000e-02, 3.54767000e-02, 3.42542000e-02, 3.30317000e-02,
               3.18092000e-02, 3.05868000e-02, 3.00000000e-02, 2.97671000e-02,
               2.94865000e-02, 2.92060000e-02, 2.89254000e-02, 2.86448000e-02,
               2.83643000e-02, 2.80837000e-02, 2.78031000e-02, 2.75226000e-02,
               2.72420000e-02, 2.69614000e-02, 2.66809000e-02, 2.64003000e-02,
               2.64000000e-02, 2.52587000e-02, 2.40599000e-02, 2.40000000e-02,
               3.00000000e-02, 3.50000000e-02, 4.00000000e-02, 5.00000000e-02,
               4.00000000e-02, 2.07120000e-02, 1.76500000e-02, 2.17931000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 5.00000000e-02,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 2.92640000e-02,
               2.40000000e-02, 2.40000000e-02, 2.40000000e-02, 2.40000000e-02,
               2.03319000e-02, 1.71500000e-02 ],
             [ 2.90000000e-02, 3.81667000e-02, 4.00000000e-02, 5.56757000e-02,
               6.00000000e-02, 5.46248000e-02, 4.38745000e-02, 3.31242000e-02,
               2.23750000e-02, 1.70000000e-02, 1.87858000e-02, 2.23571000e-02,
               2.59286000e-02, 2.95000000e-02, 3.00000000e-02, 2.98911000e-02,
               2.87452000e-02, 2.81723000e-02, 2.75993000e-02, 2.70264000e-02,
               2.64535000e-02, 2.58805000e-02, 2.53076000e-02, 2.47346000e-02,
               2.41617000e-02, 2.38500000e-02, 2.68158000e-02, 3.01000000e-02,
               2.89000000e-02, 2.93678000e-02, 3.02075000e-02, 3.10473000e-02,
               3.12744000e-02, 3.10579000e-02, 3.08413000e-02, 3.06248000e-02,
               3.04082000e-02, 3.01917000e-02, 2.99751000e-02, 2.97586000e-02,
               2.95421000e-02, 2.93255000e-02, 2.91090000e-02, 2.89000000e-02,
               2.98518000e-02, 3.24241000e-02, 3.37000000e-02, 3.44200000e-02,
               3.56022000e-02, 3.67845000e-02, 3.79666000e-02, 3.91488000e-02,
               4.00000000e-02, 4.08859000e-02, 4.29460000e-02, 4.50061000e-02,
               4.70662000e-02, 4.91263000e-02, 5.11863000e-02, 5.32464000e-02,
               5.53064000e-02, 5.73665000e-02, 5.94266000e-02, 6.14867000e-02,
               6.35468000e-02, 6.40000000e-02, 6.97931000e-02, 8.18621000e-02,
               9.39310000e-02, 1.06000000e-01, 4.00000000e-02, 2.60000000e-02,
               3.06668000e-02, 3.53336000e-02, 4.00000000e-02, 4.00000000e-02,
               3.66992000e-02, 3.54767000e-02, 3.42542000e-02, 3.30317000e-02,
               3.18092000e-02, 3.05868000e-02, 3.00000000e-02, 3.00841000e-02,
               3.01854000e-02, 3.02867000e-02, 3.03880000e-02, 3.04894000e-02,
               3.05907000e-02, 3.06920000e-02, 3.07933000e-02, 3.08946000e-02,
               3.09959000e-02, 3.10973000e-02, 3.11986000e-02, 3.12999000e-02,
               3.13000000e-02, 3.01587000e-02, 2.89599000e-02, 2.89000000e-02,
               3.00000000e-02, 3.50000000e-02, 4.00000000e-02, 4.00000000e-02,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 4.00000000e-02,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 3.25519000e-02,
               2.89000000e-02, 2.72182000e-02, 2.46229000e-02, 2.40000000e-02,
               2.29290000e-02, 2.20000000e-02 ],
             [ 2.90000000e-02, 3.81667000e-02, 4.00000000e-02, 5.56757000e-02,
               6.00000000e-02, 5.46248000e-02, 4.38745000e-02, 3.31242000e-02,
               2.23750000e-02, 1.70000000e-02, 1.87858000e-02, 2.23571000e-02,
               2.59286000e-02, 2.95000000e-02, 3.00000000e-02, 2.98911000e-02,
               2.87452000e-02, 2.81723000e-02, 2.75993000e-02, 2.70264000e-02,
               2.64535000e-02, 2.58805000e-02, 2.53076000e-02, 2.47346000e-02,
               2.41617000e-02, 2.38500000e-02, 2.68158000e-02, 3.01000000e-02,
               2.89000000e-02, 2.93678000e-02, 3.02075000e-02, 3.10473000e-02,
               3.12744000e-02, 3.10579000e-02, 3.08413000e-02, 3.06248000e-02,
               3.04082000e-02, 3.01917000e-02, 2.99751000e-02, 2.97586000e-02,
               2.95421000e-02, 2.93255000e-02, 2.91090000e-02, 2.89000000e-02,
               2.98518000e-02, 3.24241000e-02, 3.37000000e-02, 3.44200000e-02,
               3.56022000e-02, 3.67845000e-02, 3.79666000e-02, 3.91488000e-02,
               4.00000000e-02, 3.95017000e-02, 3.83429000e-02, 3.71841000e-02,
               3.60252000e-02, 3.48664000e-02, 3.37077000e-02, 3.25489000e-02,
               3.13901000e-02, 3.02313000e-02, 2.90725000e-02, 2.79137000e-02,
               2.67549000e-02, 2.65000000e-02, 3.74655000e-02, 6.03103000e-02,
               8.31552000e-02, 1.06000000e-01, 4.00000000e-02, 2.60000000e-02,
               3.06668000e-02, 3.53336000e-02, 4.00000000e-02, 4.00000000e-02,
               3.66992000e-02, 3.54767000e-02, 3.42542000e-02, 3.30317000e-02,
               3.18092000e-02, 3.05868000e-02, 3.00000000e-02, 3.00841000e-02,
               3.01854000e-02, 3.02867000e-02, 3.03880000e-02, 3.04894000e-02,
               3.05907000e-02, 3.06920000e-02, 3.07933000e-02, 3.08946000e-02,
               3.09959000e-02, 3.10973000e-02, 3.11986000e-02, 3.12999000e-02,
               3.13000000e-02, 3.01587000e-02, 2.89599000e-02, 2.89000000e-02,
               3.00000000e-02, 3.50000000e-02, 4.00000000e-02, 4.00000000e-02,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 4.00000000e-02,
               4.00000000e-02, 2.49407000e-02, 2.25500000e-02, 2.34961000e-02,
               2.40000000e-02, 2.88000000e-02, 4.00000000e-02, 3.25519000e-02,
               2.89000000e-02, 2.72182000e-02, 2.46229000e-02, 2.40000000e-02,
               2.29290000e-02, 2.20000000e-02 ]]

# B-field z edges
bfield_z = [ -10.5,      59.622,     63.022,     63.888,     67.288,
             68.154,     71.554,     72.42,      75.82,      76.686,
             80.086,     80.952,     84.352,     153.175,    162.624999,
             269.584247, 283.884247, 285.244742, 299.544742 ]

# B-field values
bfield_vals = [ 0.,         -0.37634956,  0.,         -0.37634956,  0.,
               -0.37634956,  0.,         -0.37634956,  0.,         -0.37634956,
                0.,         -0.37634956,  0.,          0.81243718,  0.,
               -2.42516915,  0.,         -2.42516915,  0.]

# Bprime z edges
bprime_z = [ -10.5,      22.965,     29.335,     32.05,      37.55,
             38.55,      44.05,      46.965,     53.335,     167.852999,
             171.252999, 194.089999, 198.889999, 225.989999, 230.789999,
             260.516999, 263.916999, 264.283999, 267.683999 ]

# Bprime values
bprime_vals = [0.,  197.69828834, 0., -198.815642,
               0., -198.815642,   0.,  199.0995432,
               0.,  -76.79520332, 0.,   40.58236262,
               0.,   -7.31680015, 0.,  152.15416786,
               0.,  152.15416786, 0. ]


def patch(path, ao):
    if "FASER_2024_I2855783/d32" not in path:
        return ao
    # when we hit the last object, append some more stuff
    name = '/REF/FASER_2024_I2855783/_aux_'
    aos = [ ao ]

    aos.append( yoda.BinnedEstimate2D([1,2,3,4], aper_edges.edges()[1:-1], name+'aper') )
    for b in aos[-1].bins():
        idxy = aper_edges.index(b.yMid())
        b.setVal(aper_vals[ b.xEdge()-1 ][ idxy-1 ])
        if idxy == aper_edges.numBins():
            aos[-1].binAt(b.xEdge(), 300.).setVal(aper_vals[b.xEdge()-1][idxy])

    aos.append( yoda.BinnedEstimate1D(bfield_z, name+'bfield') )
    aos[-1].bin(len(bfield_vals)).setVal(bfield_vals[-1])
    for b in aos[-1].bins():
        b.setVal(bfield_vals[b.index()-1])


    aos.append( yoda.BinnedEstimate1D(bprime_z, name+'bprime') )
    aos[-1].bin(len(bprime_vals)).setVal(bprime_vals[-1])
    for b in aos[-1].bins():
        b.setVal(bprime_vals[b.index()-1])

    aos.append( yoda.BinnedEstimate1D([29.842, 38.019, 53.814, 163.9895, 230.98], name+'kickers') )
    kick_vals = [ 5.685250e-06, 5.685250e-06, 5.685250e-06, -1.776600e-05, -1.995092e-05 ]
    aos[-1].bin(len(kick_vals)).setVal(kick_vals[-1])
    for b in aos[-1].bins():
        b.setVal(kick_vals[b.index()-1])

    for pid, (edges, vals) in cdf_energies.items():
        aos.append( yoda.BinnedEstimate1D(edges, f'{name}energy_{pid}') )
        aos[-1].bin(0).setVal(vals[0])
        for b in aos[-1].bins():
            b.setVal(vals[b.index()])

    aos.append( yoda.BinnedEstimate2D(hadrons, [14, -14], name+'decay_branchings') )
    for had, brs in decay_branchings.items():
        for idx, br in enumerate(brs):
            if br < 0.0:  continue
            vid = -14 if idx else 14
            aos[-1].binAt(had, vid).setVal(br)

    aos.append( yoda.BinnedEstimate2D(neutrinos, energies.edges()[1:-1], name+'xsint') )
    for b in aos[-1].bins():
        idxy = energies.index(b.yMid())
        b.setVal(xsint[ b.xEdge() ][ idxy-1 ])
        if idxy == energies.numBins():
            aos[f'{name}xsint'].binAt( b.xEdge(), 13.6e3 ).setVal(xsint[ b.xEdge() ][ idxy ])

    return aos
