/*
 * Decompiled with CFR 0.152.
 */
package org.asamk.signal.manager.storage.accounts;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.asamk.signal.manager.api.Pair;
import org.asamk.signal.manager.api.ServiceEnvironment;
import org.asamk.signal.manager.storage.SignalAccount;
import org.asamk.signal.manager.storage.Utils;
import org.asamk.signal.manager.storage.accounts.AccountsStorage;
import org.asamk.signal.manager.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.whispersystems.signalservice.api.push.ServiceId;

public class AccountsStore {
    private static final int MINIMUM_STORAGE_VERSION = 1;
    private static final int CURRENT_STORAGE_VERSION = 2;
    private static final Logger logger = LoggerFactory.getLogger(AccountsStore.class);
    private final ObjectMapper objectMapper = Utils.createStorageObjectMapper();
    private final File dataPath;
    private final String serviceEnvironment;
    private final AccountLoader accountLoader;

    public AccountsStore(File dataPath, ServiceEnvironment serviceEnvironment, AccountLoader accountLoader) throws IOException {
        this.dataPath = dataPath;
        this.serviceEnvironment = this.getServiceEnvironmentString(serviceEnvironment);
        this.accountLoader = accountLoader;
        if (!this.getAccountsFile().exists()) {
            this.createInitialAccounts();
        }
    }

    public synchronized Set<String> getAllNumbers() throws IOException {
        return this.readAccounts().stream().map(AccountsStorage.Account::number).filter(Objects::nonNull).collect(Collectors.toSet());
    }

    public synchronized Set<AccountsStorage.Account> getAllAccounts() throws IOException {
        return this.readAccounts().stream().filter(a -> a.environment() == null || this.serviceEnvironment.equals(a.environment())).filter(a -> a.number() != null).collect(Collectors.toSet());
    }

    public synchronized String getPathByNumber(String number) throws IOException {
        return this.readAccounts().stream().filter(a -> a.environment() == null || this.serviceEnvironment.equals(a.environment())).filter(a -> number.equals(a.number())).map(AccountsStorage.Account::path).findFirst().orElse(null);
    }

    public synchronized String getPathByAci(ServiceId.ACI aci) throws IOException {
        return this.readAccounts().stream().filter(a -> a.environment() == null || this.serviceEnvironment.equals(a.environment())).filter(a -> aci.toString().equals(a.uuid())).map(AccountsStorage.Account::path).findFirst().orElse(null);
    }

    public synchronized void updateAccount(String path, String number, ServiceId.ACI aci) {
        this.updateAccounts(accounts -> accounts.stream().map(a -> {
            if (a.environment() != null && !this.serviceEnvironment.equals(a.environment())) {
                return a;
            }
            if (path.equals(a.path())) {
                return new AccountsStorage.Account(a.path(), this.serviceEnvironment, number, aci == null ? null : aci.toString());
            }
            if (number != null && number.equals(a.number())) {
                return new AccountsStorage.Account(a.path(), a.environment(), null, a.uuid());
            }
            if (aci != null && aci.toString().equals(a.toString())) {
                return new AccountsStorage.Account(a.path(), a.environment(), a.number(), null);
            }
            return a;
        }).toList());
    }

    public synchronized String addAccount(String number, ServiceId.ACI aci) {
        String accountPath = this.generateNewAccountPath();
        AccountsStorage.Account account = new AccountsStorage.Account(accountPath, this.serviceEnvironment, number, aci == null ? null : aci.toString());
        this.updateAccounts(accounts -> {
            Stream<AccountsStorage.Account> existingAccounts = accounts.stream().map(a -> {
                if (a.environment() != null && !this.serviceEnvironment.equals(a.environment())) {
                    return a;
                }
                if (number != null && number.equals(a.number())) {
                    return new AccountsStorage.Account(a.path(), a.environment(), null, a.uuid());
                }
                if (aci != null && aci.toString().equals(a.uuid())) {
                    return new AccountsStorage.Account(a.path(), a.environment(), a.number(), null);
                }
                return a;
            });
            return Stream.concat(existingAccounts, Stream.of(account)).toList();
        });
        return accountPath;
    }

    public void removeAccount(String accountPath) {
        this.updateAccounts(accounts -> accounts.stream().filter(a -> a.environment() != null && !this.serviceEnvironment.equals(a.environment()) || !a.path().equals(accountPath)).toList());
    }

    private String generateNewAccountPath() {
        return new Random().ints(100000, 1000000).mapToObj(String::valueOf).filter(n -> !new File(this.dataPath, (String)n).exists() && !new File(this.dataPath, n + ".d").exists()).findFirst().get();
    }

    private File getAccountsFile() {
        return new File(this.dataPath, "accounts.json");
    }

    private void createInitialAccounts() throws IOException {
        Set<String> legacyAccountPaths = this.getLegacyAccountPaths();
        AccountsStorage accountsStorage = new AccountsStorage(legacyAccountPaths.stream().map(number -> new AccountsStorage.Account((String)number, null, (String)number, null)).toList(), 2);
        IOUtils.createPrivateDirectories(this.dataPath);
        File fileName = this.getAccountsFile();
        if (!fileName.exists()) {
            IOUtils.createPrivateFile(fileName);
        }
        Pair<FileChannel, FileLock> pair = AccountsStore.openFileChannel(this.getAccountsFile());
        try (FileChannel fileChannel = pair.first();
             FileLock lock = pair.second();){
            this.saveAccountsLocked(fileChannel, accountsStorage);
        }
    }

    private Set<String> getLegacyAccountPaths() {
        File[] files = this.dataPath.listFiles();
        if (files == null) {
            return Set.of();
        }
        return Arrays.stream(files).filter(File::isFile).map(File::getName).filter(file -> PhoneNumberUtil.getInstance().isPossibleNumber((CharSequence)file, null)).collect(Collectors.toSet());
    }

    private List<AccountsStorage.Account> readAccounts() throws IOException {
        Pair<FileChannel, FileLock> pair = AccountsStore.openFileChannel(this.getAccountsFile());
        try (FileChannel fileChannel = pair.first();){
            List<AccountsStorage.Account> list;
            block18: {
                AccountsStorage storage;
                FileLock lock;
                block16: {
                    List<AccountsStorage.Account> list2;
                    block17: {
                        lock = pair.second();
                        try {
                            int accountsVersion;
                            storage = this.readAccountsLocked(fileChannel);
                            int n = accountsVersion = storage.version() == null ? 1 : storage.version();
                            if (accountsVersion > 2) {
                                throw new IOException("Accounts file was created by a more recent version: " + accountsVersion);
                            }
                            if (accountsVersion < 1) {
                                throw new IOException("Accounts file was created by a no longer supported older version: " + accountsVersion);
                            }
                            if (accountsVersion >= 2) break block16;
                            list2 = this.upgradeAccountsFile(fileChannel, storage, accountsVersion).accounts();
                            if (lock == null) break block17;
                        }
                        catch (Throwable throwable) {
                            if (lock != null) {
                                try {
                                    lock.close();
                                }
                                catch (Throwable throwable2) {
                                    throwable.addSuppressed(throwable2);
                                }
                            }
                            throw throwable;
                        }
                        lock.close();
                    }
                    return list2;
                }
                list = storage.accounts();
                if (lock == null) break block18;
                lock.close();
            }
            return list;
        }
    }

    private AccountsStorage upgradeAccountsFile(FileChannel fileChannel, AccountsStorage storage, int accountsVersion) {
        try {
            List<AccountsStorage.Account> newAccounts = storage.accounts();
            if (accountsVersion < 2) {
                newAccounts = newAccounts.stream().map(a -> {
                    if (a.environment() != null) {
                        return a;
                    }
                    try (SignalAccount account = this.accountLoader.loadAccountOrNull(a.path());){
                        if (account == null || account.getServiceEnvironment() == null) {
                            AccountsStorage.Account account2 = a;
                            return account2;
                        }
                        AccountsStorage.Account account3 = new AccountsStorage.Account(a.path(), this.getServiceEnvironmentString(account.getServiceEnvironment()), a.number(), a.uuid());
                        return account3;
                    }
                }).toList();
            }
            AccountsStorage newStorage = new AccountsStorage(newAccounts, 2);
            this.saveAccountsLocked(fileChannel, newStorage);
            return newStorage;
        }
        catch (Exception e) {
            logger.warn("Failed to upgrade accounts file", (Throwable)e);
            return storage;
        }
    }

    private void updateAccounts(Function<List<AccountsStorage.Account>, List<AccountsStorage.Account>> updater) {
        try {
            Pair<FileChannel, FileLock> pair = AccountsStore.openFileChannel(this.getAccountsFile());
            try (FileChannel fileChannel = pair.first();
                 FileLock lock = pair.second();){
                AccountsStorage accountsStorage = this.readAccountsLocked(fileChannel);
                List<AccountsStorage.Account> newAccountsStorage = updater.apply(accountsStorage.accounts());
                this.saveAccountsLocked(fileChannel, new AccountsStorage(newAccountsStorage, 2));
            }
        }
        catch (IOException e) {
            logger.error("Failed to update accounts list", (Throwable)e);
        }
    }

    private AccountsStorage readAccountsLocked(FileChannel fileChannel) throws IOException {
        fileChannel.position(0L);
        InputStream inputStream = Channels.newInputStream(fileChannel);
        return (AccountsStorage)this.objectMapper.readValue(inputStream, AccountsStorage.class);
    }

    private void saveAccountsLocked(FileChannel fileChannel, AccountsStorage accountsStorage) throws IOException {
        try (ByteArrayOutputStream output = new ByteArrayOutputStream();){
            this.objectMapper.writeValue((OutputStream)output, (Object)accountsStorage);
            ByteArrayInputStream input = new ByteArrayInputStream(output.toByteArray());
            fileChannel.position(0L);
            input.transferTo(Channels.newOutputStream(fileChannel));
            fileChannel.truncate(fileChannel.position());
            fileChannel.force(false);
        }
    }

    private static Pair<FileChannel, FileLock> openFileChannel(File fileName) throws IOException {
        FileChannel fileChannel = new RandomAccessFile(fileName, "rw").getChannel();
        FileLock lock = fileChannel.tryLock();
        if (lock == null) {
            logger.info("Config file is in use by another instance, waiting\u2026");
            lock = fileChannel.lock();
            logger.info("Config file lock acquired.");
        }
        return new Pair<FileChannel, FileLock>(fileChannel, lock);
    }

    private String getServiceEnvironmentString(ServiceEnvironment serviceEnvironment) {
        return switch (serviceEnvironment) {
            default -> throw new MatchException(null, null);
            case ServiceEnvironment.LIVE -> "LIVE";
            case ServiceEnvironment.STAGING -> "STAGING";
        };
    }

    public static interface AccountLoader {
        public SignalAccount loadAccountOrNull(String var1);
    }
}

