// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/FastJets.hh"
#include "Rivet/Projections/LeptonFinder.hh"
#include "Rivet/Projections/MissingMomentum.hh"
#include "Rivet/Projections/PromptFinalState.hh"

namespace Rivet {

  /// @brief W+W- boson pair production in pp collisions at 13.6 TeV
  class CMS_2024_I2796231 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CMS_2024_I2796231);

    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections

      // The basic final-state projection:
      // all final-state particles within
      // the given eta acceptance
      const FinalState fs(Cuts::abseta < 5.0);
      const FinalState fsjet2p5(Cuts::abseta < 2.5);

      // The final-state particles declared above are clustered using FastJet with
      // the anti-kT algorithm and a jet-radius parameter 0.4
      // muons and neutrinos are excluded from the clustering
      FastJets jet2p5fs(fsjet2p5, JetAlg::ANTIKT, 0.4);
      declare(jet2p5fs, "jets2p5");

      // FinalState of prompt photons and bare muons and electrons in the event
      PromptFinalState photons(Cuts::abspid == PID::PHOTON);
      PromptFinalState bare_leps(Cuts::abspid == PID::MUON || Cuts::abspid == PID::ELECTRON);
      bare_leps.acceptTauDecays(false);

      // Dress the prompt bare leptons with prompt photons within dR < 0.1,
      // and apply some fiducial cuts on the dressed leptons
      Cut lepton_cuts = Cuts::abseta < 2.5 && Cuts::pT > 10*GeV;
      LeptonFinder dressed_leps(bare_leps, photons, 0.1, lepton_cuts);
      declare(dressed_leps, "leptons");

      // Book histograms
      book(_h_WW_njets_norm , 3, 1, 1);
      book(_h_WW_njets      , 5, 1, 1);

    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {

      if (_edges.empty()) {
        _edges = _h_WW_njets_norm->xEdges();
      }

      // Retrieve dressed leptons, sorted by pT
      Particles leptons = apply<LeptonFinder>(event, "leptons").particlesByPt();

      // Apply a lepton size requirement
      if (leptons.size() != 2) return;

      // Retrieve clustered jets, sorted by pT, with a minimum pT cut
      Jets jetsNj = apply<FastJets>(event, "jets2p5").jetsByPt(Cuts::pT > 30*GeV);

      // Remove all jets within dR < 0.4 of a dressed lepton
      idiscardIfAnyDeltaRLess(jetsNj, leptons, 0.4);

      if (leptons.size() == 2 && leptons[0].pid() * leptons[1].pid() < 0 &&
          abs(leptons[0].pid()) != abs(leptons[1].pid())) {
        FourMomentum dilCand = leptons[0].momentum() + leptons[1].momentum();
        if (dilCand.mass() > 85*GeV){
          double ptlmax = leptons[0].pT(); double ptlmin = leptons[1].pT();
          if (ptlmax < ptlmin) {
            ptlmax = leptons[1].pT(); ptlmin = leptons[0].pT();
          }
          if(ptlmax > 25*GeV && ptlmin > 20*GeV) {
            _h_WW_njets      ->fill(_edges[ min((int)jetsNj.size(), 2) ]);
            _h_WW_njets_norm ->fill(_edges[ min((int)jetsNj.size(), 2) ]);
          } // ptlmax > 25 && ptmin > 20
        } // mll > 85
      } // different-flavor leptons

    }

    /// Normalise histograms etc., after the run
    void finalize() {

      double norm = (sumOfWeights() != 0) ? crossSection()/picobarn/sumOfWeights() : 1.0;
      normalize(_h_WW_njets_norm );
      scale(_h_WW_njets, norm);

    }

    /// @}

    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _h_WW_njets_norm ;
    BinnedHistoPtr<string> _h_WW_njets;
    vector<string> _edges;
    /// @}

  };

  RIVET_DECLARE_PLUGIN(CMS_2024_I2796231);

}
