/*
 *  Copyright 2008-2013 NVIDIA Corporation
 *  Modifications Copyright© 2025 Advanced Micro Devices, Inc. All rights reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include <thrust/detail/preprocessor.h>

#include <string>

#include "test_param_fixtures.hpp"
#include "test_utils.hpp"

// clang-format off
TEST(PreprocessorTest, test_pp_stringize)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(int))
  , "int"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(hello world))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(hello  world))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE( hello  world))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(hello  world ))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE( hello  world ))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(hello
                                    world))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE("hello world"))
  , "\"hello world\""
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE('hello world'))
  , "'hello world'"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE($%!&<->))
  , "$%!&<->"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE($%!&""<->))
  , "$%!&\"\"<->"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_STRINGIZE))
  , "THRUST_PP_STRINGIZE"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_STRINGIZE(int)))
  , "\"int\""
  );
}

TEST(PreprocessorTest, test_pp_cat2)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(i, nt)))
  , "int"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello, world)))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello , world)))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2( hello, world)))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello,  world)))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello, world )))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello,
                                                   world )))
  , "helloworld"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(hello world, from thrust!)))
  , "hello worldfrom thrust!"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_CAT2(-, >)))
  , "->"
  );
}

#define THRUST_TEST_PP_EXPAND_TARGET() success

#define THRUST_TEST_PP_EXPAND_ARGS() ()

TEST(PreprocessorTest, test_pp_expand)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(int)))
  , "int"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(hello world)))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(hello  world)))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND( hello  world)))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(hello  world )))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND( hello  world )))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(hello
                                    world)))
  , "hello world"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND("hello world")))
  , "\"hello world\""
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND('hello world')))
  , "'hello world'"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND($%!&<->)))
  , "$%!&<->"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND($%!&""<->)))
  , "$%!&\"\"<->"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(THRUST_PP_EXPAND)))
  , "THRUST_PP_EXPAND"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(THRUST_PP_EXPAND(int))))
  , "int"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(
      THRUST_PP_CAT2(THRUST_TEST_, PP_EXPAND_TARGET)()
    )))
  , "success"
  );

  ASSERT_EQ(
    std::string(THRUST_PP_STRINGIZE(THRUST_PP_EXPAND(
      THRUST_TEST_PP_EXPAND_TARGET THRUST_TEST_PP_EXPAND_ARGS()
    )))
  , "success"
  );
}

#undef THRUST_TEST_PP_EXPAND_TARGET

#undef THRUST_TEST_PP_EXPAND_ARGS

TEST(PreprocessorTest, test_pp_arity)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());

  ASSERT_EQ(
    THRUST_PP_ARITY()
  , 0
  );

  /* This bash script was used to generate these tests:

    for arity in {0..62}
    do
      echo "  ASSERT_EQ("
      echo "    THRUST_PP_ARITY("
      echo "      `bash -c \"echo {0..${arity}} | tr ' ' ,\"`"
      echo "    )"
      echo "  , $((${arity} + 1))"
      echo "  );"
      echo
    done
  */

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0
    )
  , 1
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1
    )
  , 2
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2
    )
  , 3
  );
ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3
    )
  , 4
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4
    )
  , 5
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5
    )
  , 6
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6
    )
  , 7
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7
    )
  , 8
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8
    )
  , 9
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9
    )
  , 10
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10
    )
  , 11
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11
    )
  , 12
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12
    )
  , 13
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13
    )
  , 14
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14
    )
  , 15
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15
    )
  , 16
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
    )
  , 17
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17
    )
  , 18
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18
    )
  , 19
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19
    )
  , 20
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20
    )
  , 21
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21
    )
  , 22
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22
    )
  , 23
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23
    )
  , 24
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24
    )
  , 25
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25
    )
  , 26
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26
    )
  , 27
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27
    )
  , 28
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28
    )
  , 29
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29
    )
  , 30
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30
    )
  , 31
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31
    )
  , 32
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32
    )
  , 33
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33
    )
  , 34
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34
    )
  , 35
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35
    )
  , 36
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36
    )
  , 37
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37
    )
  , 38
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38
    )
  , 39
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39
    )
  , 40
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40
    )
  , 41
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41
    )
  , 42
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42
    )
  , 43
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43
    )
  , 44
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44
    )
  , 45
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45
    )
  , 46
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46
    )
  , 47
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47
    )
  , 48
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48
    )
  , 49
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49
    )
  , 50
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50
    )
  , 51
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51
    )
  , 52
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52
    )
  , 53
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53
    )
  , 54
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54
    )
  , 55
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55
    )
  , 56
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56
    )
  , 57
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57
    )
  , 58
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58
    )
  , 59
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59
    )
  , 60
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60
    )
  , 61
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61
    )
  , 62
  );

  ASSERT_EQ(
    THRUST_PP_ARITY(
      0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59,60,61,62
    )
  , 63
  );
}

#define THRUST_TEST_PP_DISPATCH_PLUS(...)                                     \
  THRUST_PP_DISPATCH(THRUST_TEST_PP_DISPATCH_PLUS, __VA_ARGS__)               \
  /**/
#define THRUST_TEST_PP_DISPATCH_PLUS0()        0
#define THRUST_TEST_PP_DISPATCH_PLUS1(x)       x
#define THRUST_TEST_PP_DISPATCH_PLUS2(x, y)    x + y
#define THRUST_TEST_PP_DISPATCH_PLUS3(x, y, z) x + y + z

TEST(PreprocessorTest, test_pp_dispatch)
{
  SCOPED_TRACE(testing::Message() << "with device_id= " << test::set_device_from_ctest());

  ASSERT_EQ(
    THRUST_TEST_PP_DISPATCH_PLUS()
  , 0
  );

  ASSERT_EQ(
    THRUST_TEST_PP_DISPATCH_PLUS(0)
  , 0
  );

  ASSERT_EQ(
    THRUST_TEST_PP_DISPATCH_PLUS(1, 2)
  , 3
  );

  ASSERT_EQ(
    THRUST_TEST_PP_DISPATCH_PLUS(1, 2, 3)
  , 6
  );
}
// clang-format on

#undef THRUST_TEST_PP_DISPATCH_PLUS
#undef THRUST_TEST_PP_DISPATCH_PLUS0
#undef THRUST_TEST_PP_DISPATCH_PLUS1
#undef THRUST_TEST_PP_DISPATCH_PLUS2
#undef THRUST_TEST_PP_DISPATCH_PLUS3
