# Go implementation of in-toto
[![build](https://github.com/in-toto/in-toto-golang/workflows/build/badge.svg)](https://github.com/in-toto/in-toto-golang/actions?query=workflow%3Abuild) [![Coverage Status](https://coveralls.io/repos/github/in-toto/in-toto-golang/badge.svg)](https://coveralls.io/github/in-toto/in-toto-golang) [![PkgGoDev](https://pkg.go.dev/badge/github.com/in-toto/in-toto-golang)](https://pkg.go.dev/github.com/in-toto/in-toto-golang) [![Go Report Card](https://goreportcard.com/badge/github.com/in-toto/in-toto-golang)](https://goreportcard.com/report/github.com/in-toto/in-toto-golang)


Go implementation of the
[in-toto specification](https://github.com/in-toto/docs/blob/master/in-toto-spec.md).

## Docs

To read the documentation along with some examples, run:

```bash
godoc -http :8080
```

and navigate to `localhost:8080/pkg/github.com/in-toto/in-toto-golang/`

Alternatively, you can use [pkg.go.dev](https://pkg.go.dev/github.com/in-toto/in-toto-golang).

## Example

A very simple example, just to help you starting:

```go
package main

import (
	"time"
	toto "github.com/in-toto/in-toto-golang/in_toto"
)

func main() {
	t := time.Now()
	t = t.Add(30 * 24 * time.Hour)

	var keys = make(map[string]toto.Key)

	var metablock = toto.Metablock{
		Signed: toto.Layout{
			Type: "layout",
			Expires:  t.Format("2006-01-02T15:04:05Z"),
			Steps: []toto.Step{},
			Inspect: []toto.Inspection{},
			Keys:  keys,
		},
	}

	var key toto.Key

	key.LoadKey("keys/alice", "rsassa-pss-sha256", []string{"sha256", "sha512"})

	metablock.Sign(key)

	metablock.Dump("root.layout")
}
```

## Building

Download the source, run `make build`.

## CLI

The CLI reference can be found in the autogenerated [docs](doc/in-toto.md).

## Integration with SPIFFE/SPIRE

This implementation of in-toto has been integrated with SPIFFE/SPIRE. The
integration is made possible by
[ITE-7](https://github.com/in-toto/ITE/blob/master/ITE/7/README.adoc), an
enhancement that adds support for X.509 signing to in-toto.

### Running the Demo

To run the demo, pull down the source code, install Go, and run
`make test-verify`. This will use openssl to generate a certificate chain.

To run the demo using SPIRE, pull down the source code, install Go and Docker,
and run `make test-spiffe-verify`.

SPIFFE compliant Leaf certificates are generated with SVIDs corresponding to
functionaries. These certificates are consumed by in-toto to sign link metadata
and the layout policy.

During the in-toto verification process, certificate constraints are checked to
ensure the build step link meta-data was signed with the correct SVID.

### Layout Certificate Constraints

Currently the following constraints supported:

```json
{
  "cert_constraints": [{
    "common_name": "write-code.example.com",
      "dns_names": [
        ""
      ],
      "emails": [
        ""
      ],
      "organizations": [
        "*"
      ],
      "roots": [
        "*"
      ],
      "uris": [
        "spiffe://example.com/write-code"
      ]
  }, {
    "uris": [],
    "common_names": ["Some User"]
  }]
}
```

## Not (yet) supported

This golang implementation was focused on verification on admission controllers
and kubectl plugins. As such, it focused on providing a strong, auditable set
of core functions rather than a broad and (possibly) unstable feature set. In
other words, we believe that the current feature set is stable enough for
production use.

If any of these features are necessary for your use case please let us know and
we will try to provide them as soon as possible. Alternatively we welcome pull
requests with feature additions!

* [GPG keys](https://github.com/in-toto/in-toto-golang/issues/26)
