// Syd: rock-solid application kernel
// src/ioctl.rs: ioctl(2) request decoder
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libc::c_ulong;
use libseccomp::ScmpArch;

use crate::{confine::SCMP_ARCH, hash::SydHashMap};

/// This type represents an ioctl(2) request.
pub type Ioctl = c_ulong;

/// This type represents an ioctl(2) list.
pub type IoctlList = &'static [(&'static str, Ioctl)];

// Include auto-generated ioctl(2) requests.
include!("ioctl/ioctls_aarch64.rs");
include!("ioctl/ioctls_arm.rs");
include!("ioctl/ioctls_loongarch64.rs");
include!("ioctl/ioctls_m68k.rs");
include!("ioctl/ioctls_mips.rs");
include!("ioctl/ioctls_mips64.rs");
include!("ioctl/ioctls_mips64n32.rs");
include!("ioctl/ioctls_mipsel.rs");
include!("ioctl/ioctls_mipsel64.rs");
include!("ioctl/ioctls_mipsel64n32.rs");
include!("ioctl/ioctls_ppc.rs");
include!("ioctl/ioctls_ppc64.rs");
include!("ioctl/ioctls_ppc64le.rs");
include!("ioctl/ioctls_riscv64.rs");
include!("ioctl/ioctls_s390.rs");
include!("ioctl/ioctls_s390x.rs");
include!("ioctl/ioctls_x32.rs");
include!("ioctl/ioctls_x86.rs");
include!("ioctl/ioctls_x8664.rs");

const ARCH_TABLES: &[(ScmpArch, IoctlList)] = &[
    (ScmpArch::Aarch64, IOCTL_ARCH_AARCH64),
    (ScmpArch::Arm, IOCTL_ARCH_ARM),
    (ScmpArch::Loongarch64, IOCTL_ARCH_LOONGARCH64),
    (ScmpArch::M68k, IOCTL_ARCH_M68K),
    (ScmpArch::Mips, IOCTL_ARCH_MIPS),
    (ScmpArch::Mips64, IOCTL_ARCH_MIPS64),
    (ScmpArch::Mips64N32, IOCTL_ARCH_MIPS64N32),
    (ScmpArch::Mipsel, IOCTL_ARCH_MIPSEL),
    (ScmpArch::Mipsel64, IOCTL_ARCH_MIPSEL64),
    (ScmpArch::Mipsel64N32, IOCTL_ARCH_MIPSEL64N32),
    (ScmpArch::Ppc, IOCTL_ARCH_PPC),
    (ScmpArch::Ppc64, IOCTL_ARCH_PPC64),
    (ScmpArch::Ppc64Le, IOCTL_ARCH_PPC64LE),
    (ScmpArch::Riscv64, IOCTL_ARCH_RISCV64),
    (ScmpArch::S390, IOCTL_ARCH_S390),
    (ScmpArch::S390X, IOCTL_ARCH_S390X),
    (ScmpArch::X32, IOCTL_ARCH_X32),
    (ScmpArch::X86, IOCTL_ARCH_X86),
    (ScmpArch::X8664, IOCTL_ARCH_X8664),
];

/// This type represents an ioctl(2) names map.
pub type IoctlNamesMap = SydHashMap<Ioctl, Vec<&'static str>>;

/// This type represents an ioctl(2) values map.
pub type IoctlValueMap = SydHashMap<&'static str, Ioctl>;

/// This type represents ioctl(2) names maps per architecture.
pub type IoctlArchNamesMap = SydHashMap<ScmpArch, IoctlNamesMap>;

/// This type represents ioctl(2) value maps per architecture.
pub type IoctlArchValueMap = SydHashMap<ScmpArch, IoctlValueMap>;

/// This structure represents ioctl maps.
///
/// It offers an API to query ioctls by name and by value.
pub struct IoctlMap {
    /// Ioctl names map per architecture.
    pub names_map: IoctlArchNamesMap,
    /// Ioctl value map per architecture.
    pub value_map: IoctlArchValueMap,
}

impl IoctlMap {
    /// Initialize a new IoctlMap.
    ///
    /// If `native` is true only architectures native
    /// to the current architecture are initialized,
    /// e.g. for x86-64, this is x86-64, x32, and x86.
    pub fn new(native: bool) -> Self {
        let mut v2n_outer = IoctlArchNamesMap::default();
        let mut n2v_outer = IoctlArchValueMap::default();

        for &(arch, table) in ARCH_TABLES {
            if native && !SCMP_ARCH.contains(&arch) {
                continue;
            }

            let v2n = v2n_outer.entry(arch).or_insert_with(IoctlNamesMap::default);
            let n2v = n2v_outer.entry(arch).or_insert_with(IoctlValueMap::default);

            for &(name, val) in table {
                let val = Ioctl::from(val);

                v2n.entry(val).or_default().push(name);

                // Keep first mapping for a given name,
                // if conflicting values exist.
                n2v.entry(name).or_insert(val);
            }

            // Stable per-arch normalization.
            for names in v2n.values_mut() {
                names.sort_unstable();
                names.dedup();
            }
        }

        Self {
            names_map: v2n_outer,
            value_map: n2v_outer,
        }
    }

    /// Return the architecture specific Ioctl map.
    pub fn get_map(&self, arch: ScmpArch) -> Option<&IoctlNamesMap> {
        self.names_map.get(&arch)
    }

    /// Return symbol names for the given Ioctl.
    pub fn get_names(&self, value: Ioctl, arch: ScmpArch) -> Option<Vec<&'static str>> {
        self.names_map
            .get(&arch)
            .and_then(|inner| inner.get(&value))
            .cloned()
    }

    /// Return Ioctl request number for the given symbol name.
    pub fn get_value(&self, name: &str, arch: ScmpArch) -> Option<Ioctl> {
        self.value_map
            .get(&arch)
            .and_then(|inner| inner.get(name))
            .copied()
    }
}
