/* SPDX-License-Identifier: GPL-2.0-only */

#include <console/console.h>
#include <soc/addressmap.h>
#include <soc/pmc.h>
#include <soc/sdram.h>

/*
 * This function reads SDRAM parameters from the common BCT format and
 * writes them into PMC scratch registers (where the BootROM expects them
 * on LP0 resume).
 */
void sdram_lp0_save_params(const struct sdram_params *sdram)
{
	struct tegra_pmc_regs *pmc = (void *)TEGRA_PMC_BASE;

#define pack(src, src_bits, dst, dst_bits) { \
	_Static_assert((1 ? src_bits) >= (0 ? src_bits) && (1 ? dst_bits) >= \
		(0 ? dst_bits), "byte range flipped (must be MSB:LSB)"); \
	_Static_assert((1 ? src_bits) - (0 ? src_bits) == (1 ? dst_bits) - \
		(0 ? dst_bits), "src and dst byte range lengths differ"); \
	u32 mask = 0xffffffff >> (31 - ((1 ? src_bits) - (0 ? src_bits))); \
	dst &= ~(mask << (0 ? dst_bits)); \
	dst |= ((src >> (0 ? src_bits)) & mask) << (0 ? dst_bits); \
}

#define s(param, src_bits, pmcreg, dst_bits) \
	pack(sdram->param, src_bits, pmc->pmcreg, dst_bits)

#define c(value, pmcreg, dst_bits) \
	pack(value, (1 ? dst_bits) - (0 ? dst_bits) : 0, pmc->pmcreg, dst_bits)

/* 32 bits version of s macro */
#define s32(param, pmcreg) pmc->pmcreg = sdram->param

/* 32 bits version c macro */
#define c32(value, pmcreg) pmc->pmcreg = value

	s(EmcClockSource, 7:0, scratch6, 15:8);
	s(EmcClockSourceDll, 7:0, scratch6, 23:16);
	s(EmcClockSource, 31:29, scratch6, 26:24);
	s(EmcClockSourceDll, 31:29, scratch6, 29:27);
	s(EmcClockSourceDll, 11:10, scratch6, 31:30);
	s(ClkRstControllerPllmMisc2Override, 9:8, scratch7, 1:0);
	s(ClkRstControllerPllmMisc2Override, 2:1, scratch7, 3:2);
	s(EmcZqCalLpDdr4WarmBoot, 31:30, scratch7, 5:4);
	s(EmcClockSource, 15:15, scratch7, 6:6);
	s(EmcClockSource, 26:26, scratch7, 7:7);
	s(EmcClockSource, 20:20, scratch7, 8:8);
	s(EmcClockSource, 19:19, scratch7, 9:9);
	s(ClkRstControllerPllmMisc2Override, 13:13, scratch7, 10:10);
	s(ClkRstControllerPllmMisc2Override, 12:12, scratch7, 11:11);
	s(ClkRstControllerPllmMisc2Override, 11:11, scratch7, 12:12);
	s(ClkRstControllerPllmMisc2Override, 10:10, scratch7, 13:13);
	s(ClkRstControllerPllmMisc2Override, 5:5, scratch7, 14:14);
	s(ClkRstControllerPllmMisc2Override, 4:4, scratch7, 15:15);
	s(ClkRstControllerPllmMisc2Override, 3:3, scratch7, 16:16);
	s(ClkRstControllerPllmMisc2Override, 0:0, scratch7, 17:17);
	s(EmcZqCalLpDdr4WarmBoot, 1:0, scratch7, 19:18);
	s(EmcZqCalLpDdr4WarmBoot, 4:4, scratch7, 20:20);
	s(EmcOdtWrite, 5:0, scratch7, 26:21);
	s(EmcOdtWrite, 11:8, scratch7, 30:27);
	s(EmcOdtWrite, 31:31, scratch7, 31:31);
	s(EmcFdpdCtrlCmdNoRamp, 0:0, scratch13, 30:30);
	s(EmcCfgPipeClk, 0:0, scratch13, 31:31);
	s(McEmemArbMisc2, 0:0, scratch14, 30:30);
	s(McDaCfg0, 0:0, scratch14, 31:31);
	s(EmcQRst, 6:0, scratch15, 26:20);
	s(EmcQRst, 20:16, scratch15, 31:27);
	s(EmcPmacroCmdTxDrv, 5:0, scratch16, 25:20);
	s(EmcPmacroCmdTxDrv, 13:8, scratch16, 31:26);
	s(EmcPmacroAutocalCfg0, 2:0, scratch17, 22:20);
	s(EmcPmacroAutocalCfg0, 10:8, scratch17, 25:23);
	s(EmcPmacroAutocalCfg0, 18:16, scratch17, 28:26);
	s(EmcPmacroAutocalCfg0, 26:24, scratch17, 31:29);
	s(EmcPmacroAutocalCfg1, 2:0, scratch18, 22:20);
	s(EmcPmacroAutocalCfg1, 10:8, scratch18, 25:23);
	s(EmcPmacroAutocalCfg1, 18:16, scratch18, 28:26);
	s(EmcPmacroAutocalCfg1, 26:24, scratch18, 31:29);
	s(EmcPmacroAutocalCfg2, 2:0, scratch19, 22:20);
	s(EmcPmacroAutocalCfg2, 10:8, scratch19, 25:23);
	s(EmcPmacroAutocalCfg2, 18:16, scratch19, 28:26);
	s(EmcPmacroAutocalCfg2, 26:24, scratch19, 31:29);
	s32(EmcCfgRsv,scratch22);
	s32(EmcAutoCalConfig, scratch23);
	s32(EmcAutoCalVrefSel0, scratch24);
	s32(EmcPmacroBrickCtrlRfu1, scratch25);
	s32(EmcPmacroBrickCtrlRfu2, scratch26);
	s32(EmcPmcScratch1, scratch27);
	s32(EmcPmcScratch2, scratch28);
	s32(EmcPmcScratch3, scratch29);
	s32(McEmemArbDaTurns, scratch30);
	s(EmcFbioSpare, 31:24, scratch58, 7:0);
	s(EmcFbioSpare, 23:16, scratch58, 15:8);
	s(EmcFbioSpare, 15:8, scratch58, 23:16);
	s(EmcFbioSpare, 7:2, scratch58, 29:24);
	s(EmcFbioSpare, 0:0, scratch58, 30:30);
	s(EmcDllCfg0, 29:0, scratch59, 29:0);
	s(EmcPmacroDdllBypass, 11:0, scratch60, 11:0);
	s(EmcPmacroDdllBypass, 27:13, scratch60, 26:12);
	s(EmcPmacroDdllBypass, 31:29, scratch60, 29:27);
	s(McEmemArbMisc0, 14:0, scratch61, 14:0);
	s(McEmemArbMisc0, 30:16, scratch61, 29:15);
	s(EmcFdpdCtrlCmd, 16:0, scratch62, 16:0);
	s(EmcFdpdCtrlCmd, 31:20, scratch62, 28:17);
	s(EmcAutoCalConfig2, 27:0, scratch63, 27:0);
	s(EmcBurstRefreshNum, 3:0, scratch63, 31:28);
	s(EmcPmacroZctrl, 27:0, scratch64, 27:0);
	s(EmcTppd, 3:0, scratch64, 31:28);
	s(EmcCfgDigDll, 10:0, scratch65, 10:0);
	s(EmcCfgDigDll, 25:12, scratch65, 24:11);
	s(EmcCfgDigDll, 27:27, scratch65, 25:25);
	s(EmcCfgDigDll, 31:30, scratch65, 27:26);
	s(EmcR2r, 3:0, scratch65, 31:28);
	s(EmcFdpdCtrlDq, 16:0, scratch66, 16:0);
	s(EmcFdpdCtrlDq, 28:20, scratch66, 25:17);
	s(EmcFdpdCtrlDq, 31:30, scratch66, 27:26);
	s(EmcW2w, 3:0, scratch66, 31:28);
	s(EmcPmacroTxPwrd4, 13:0, scratch67, 13:0);
	s(EmcPmacroTxPwrd4, 29:16, scratch67, 27:14);
	s(EmcPmacroCommonPadTxCtrl, 3:0, scratch67, 31:28);
	s(EmcPmacroTxPwrd5, 13:0, scratch68, 13:0);
	s(EmcPmacroTxPwrd5, 29:16, scratch68, 27:14);
	s(EmcPmacroDdllPwrd0, 4:0, scratch69, 4:0);
	s(EmcPmacroDdllPwrd0, 12:6, scratch69, 11:5);
	s(EmcPmacroDdllPwrd0, 20:14, scratch69, 18:12);
	s(EmcPmacroDdllPwrd0, 28:22, scratch69, 25:19);
	s(EmcPmacroDdllPwrd0, 31:30, scratch69, 27:26);
	s(EmcCfg, 4:4, scratch69, 31:31);
	s(EmcPmacroDdllPwrd1, 4:0, scratch70, 4:0);
	s(EmcPmacroDdllPwrd1, 12:6, scratch70, 11:5);
	s(EmcPmacroDdllPwrd1, 20:14, scratch70, 18:12);
	s(EmcPmacroDdllPwrd1, 28:22, scratch70, 25:19);
	s(EmcPmacroDdllPwrd1, 31:30, scratch70, 27:26);
	s(EmcCfg, 5:5, scratch70, 31:31);
	s(EmcPmacroDdllPwrd2, 4:0, scratch71, 4:0);
	s(EmcPmacroDdllPwrd2, 12:6, scratch71, 11:5);
	s(EmcPmacroDdllPwrd2, 20:14, scratch71, 18:12);
	s(EmcPmacroDdllPwrd2, 28:22, scratch71, 25:19);
	s(EmcPmacroDdllPwrd2, 31:30, scratch71, 27:26);
	s(EmcFbioCfg5, 23:20, scratch71, 31:28);
	s(EmcPmacroIbVrefDq_0, 6:0, scratch72, 6:0);
	s(EmcPmacroIbVrefDq_0, 14:8, scratch72, 13:7);
	s(EmcPmacroIbVrefDq_0, 22:16, scratch72, 20:14);
	s(EmcPmacroIbVrefDq_0, 30:24, scratch72, 27:21);
	s(EmcFbioCfg5, 15:13, scratch72, 30:28);
	s(EmcCfg, 6:6, scratch72, 31:31);
	s(EmcPmacroIbVrefDq_1, 6:0, scratch73, 6:0);
	s(EmcPmacroIbVrefDq_1, 14:8, scratch73, 13:7);
	s(EmcPmacroIbVrefDq_1, 22:16, scratch73, 20:14);
	s(EmcPmacroIbVrefDq_1, 30:24, scratch73, 27:21);
	s(EmcCfg2, 5:3, scratch73, 30:28);
	s(EmcCfg, 7:7, scratch73, 31:31);
	s(EmcPmacroIbVrefDqs_0, 6:0, scratch74, 6:0);
	s(EmcPmacroIbVrefDqs_0, 14:8, scratch74, 13:7);
	s(EmcPmacroIbVrefDqs_0, 22:16, scratch74, 20:14);
	s(EmcPmacroIbVrefDqs_0, 30:24, scratch74, 27:21);
	s(EmcCfg, 17:16, scratch74, 29:28);
	s(EmcFbioCfg5, 1:0, scratch74, 31:30);
	s(EmcPmacroIbVrefDqs_1, 6:0, scratch75, 6:0);
	s(EmcPmacroIbVrefDqs_1, 14:8, scratch75, 13:7);
	s(EmcPmacroIbVrefDqs_1, 22:16, scratch75, 20:14);
	s(EmcPmacroIbVrefDqs_1, 30:24, scratch75, 27:21);
	s(EmcFbioCfg5, 3:2, scratch75, 29:28);
	s(EmcCfg2, 27:26, scratch75, 31:30);
	s(EmcPmacroDdllShortCmd_0, 6:0, scratch76, 6:0);
	s(EmcPmacroDdllShortCmd_0, 14:8, scratch76, 13:7);
	s(EmcPmacroDdllShortCmd_0, 22:16, scratch76, 20:14);
	s(EmcPmacroDdllShortCmd_0, 30:24, scratch76, 27:21);
	s(EmcPmacroCmdPadTxCtrl, 3:2, scratch76, 29:28);
	s(EmcPmacroCmdPadTxCtrl, 7:6, scratch76, 31:30);
	s(EmcPmacroDdllShortCmd_1, 6:0, scratch77, 6:0);
	s(EmcPmacroDdllShortCmd_1, 14:8, scratch77, 13:7);
	s(EmcPmacroDdllShortCmd_1, 22:16, scratch77, 20:14);
	s(EmcPmacroDdllShortCmd_1, 30:24, scratch77, 27:21);
	s(EmcPmacroCmdPadTxCtrl, 11:10, scratch77, 29:28);
	s(EmcPmacroCmdPadTxCtrl, 15:14, scratch77, 31:30);
	s(EmcAutoCalChannel, 5:0, scratch78, 5:0);
	s(EmcAutoCalChannel, 11:8, scratch78, 9:6);
	s(EmcAutoCalChannel, 27:16, scratch78, 21:10);
	s(EmcAutoCalChannel, 31:29, scratch78, 24:22);
	s(EmcConfigSampleDelay, 6:0, scratch78, 31:25);
	s(EmcPmacroRxTerm, 5:0, scratch79, 5:0);
	s(EmcPmacroRxTerm, 13:8, scratch79, 11:6);
	s(EmcPmacroRxTerm, 21:16, scratch79, 17:12);
	s(EmcPmacroRxTerm, 29:24, scratch79, 23:18);
	s(EmcRc, 7:0, scratch79, 31:24);
	s(EmcPmacroDqTxDrv, 5:0, scratch80, 5:0);
	s(EmcPmacroDqTxDrv, 13:8, scratch80, 11:6);
	s(EmcPmacroDqTxDrv, 21:16, scratch80, 17:12);
	s(EmcPmacroDqTxDrv, 29:24, scratch80, 23:18);
	s(EmcSelDpdCtrl, 5:2, scratch80, 27:24);
	s(EmcSelDpdCtrl, 8:8, scratch80, 28:28);
	s(EmcSelDpdCtrl, 18:16, scratch80, 31:29);
	s(EmcPmacroCaTxDrv, 5:0, scratch81, 5:0);
	s(EmcPmacroCaTxDrv, 13:8, scratch81, 11:6);
	s(EmcPmacroCaTxDrv, 21:16, scratch81, 17:12);
	s(EmcPmacroCaTxDrv, 29:24, scratch81, 23:18);
	s(EmcObdly, 5:0, scratch81, 29:24);
	s(EmcObdly, 29:28, scratch81, 31:30);
	s(EmcZcalInterval, 23:10, scratch82, 13:0);
	s(EmcZcalInterval, 9:0, scratch82, 23:14);
	s(EmcPmacroCmdRxTermMode, 1:0, scratch82, 25:24);
	s(EmcPmacroCmdRxTermMode, 5:4, scratch82, 27:26);
	s(EmcPmacroCmdRxTermMode, 9:8, scratch82, 29:28);
	s(EmcPmacroCmdRxTermMode, 13:12, scratch82, 31:30);
	s(EmcDataBrlshft0, 23:0, scratch83, 23:0);
	s(EmcPmacroDataRxTermMode, 1:0, scratch83, 25:24);
	s(EmcPmacroDataRxTermMode, 5:4, scratch83, 27:26);
	s(EmcPmacroDataRxTermMode, 9:8, scratch83, 29:28);
	s(EmcPmacroDataRxTermMode, 13:12, scratch83, 31:30);
	s(EmcDataBrlshft1, 23:0, scratch84, 23:0);
	s(McEmemArbTimingRc, 7:0, scratch84, 31:24);
	s(EmcDqsBrlshft0, 23:0, scratch85, 23:0);
	s(McEmemArbRsv, 7:0, scratch85, 31:24);
	s(EmcDqsBrlshft1, 23:0, scratch86, 23:0);
	s(EmcCfgPipe2, 11:0, scratch87, 11:0);
	s(EmcCfgPipe2, 27:16, scratch87, 23:12);
	s(EmcCfgPipe1, 11:0, scratch88, 11:0);
	s(EmcCfgPipe1, 27:16, scratch88, 23:12);
	s(EmcPmacroCmdCtrl0, 5:0, scratch89, 5:0);
	s(EmcPmacroCmdCtrl0, 13:8, scratch89, 11:6);
	s(EmcPmacroCmdCtrl0, 21:16, scratch89, 17:12);
	s(EmcPmacroCmdCtrl0, 29:24, scratch89, 23:18);
	s(EmcPmacroCmdCtrl1, 5:0, scratch90, 5:0);
	s(EmcPmacroCmdCtrl1, 13:8, scratch90, 11:6);
	s(EmcPmacroCmdCtrl1, 21:16, scratch90, 17:12);
	s(EmcPmacroCmdCtrl1, 29:24, scratch90, 23:18);
	s(EmcRas, 6:0, scratch90, 30:24);
	s(EmcCfg, 8:8, scratch90, 31:31);
	s(EmcPmacroVttgenCtrl2, 23:0, scratch91, 23:0);
	s(EmcW2p, 6:0, scratch91, 30:24);
	s(EmcCfg, 9:9, scratch91, 31:31);
	s(EmcPmacroCmdPadRxCtrl, 2:0, scratch92, 2:0);
	s(EmcPmacroCmdPadRxCtrl, 5:4, scratch92, 4:3);
	s(EmcPmacroCmdPadRxCtrl, 10:8, scratch92, 7:5);
	s(EmcPmacroCmdPadRxCtrl, 22:12, scratch92, 18:8);
	s(EmcPmacroCmdPadRxCtrl, 28:24, scratch92, 23:19);
	s(EmcQSafe, 6:0, scratch92, 30:24);
	s(EmcCfg, 18:18, scratch92, 31:31);
	s(EmcPmacroDataPadRxCtrl, 2:0, scratch93, 2:0);
	s(EmcPmacroDataPadRxCtrl, 5:4, scratch93, 4:3);
	s(EmcPmacroDataPadRxCtrl, 10:8, scratch93, 7:5);
	s(EmcPmacroDataPadRxCtrl, 22:12, scratch93, 18:8);
	s(EmcPmacroDataPadRxCtrl, 28:24, scratch93, 23:19);
	s(EmcRdv, 6:0, scratch93, 30:24);
	s(EmcCfg, 21:21, scratch93, 31:31);
	s(McEmemArbDaCovers, 23:0, scratch94, 23:0);
	s(EmcRw2Pden, 6:0, scratch94, 30:24);
	s(EmcCfg, 22:22, scratch94, 31:31);
	s(EmcPmacroCmdCtrl2, 5:0, scratch95, 5:0);
	s(EmcPmacroCmdCtrl2, 13:9, scratch95, 10:6);
	s(EmcPmacroCmdCtrl2, 21:16, scratch95, 16:11);
	s(EmcPmacroCmdCtrl2, 29:24, scratch95, 22:17);
	s(EmcRfcPb, 8:0, scratch95, 31:23);
	s(EmcPmacroQuseDdllRank0_0, 10:0, scratch96, 10:0);
	s(EmcPmacroQuseDdllRank0_0, 26:16, scratch96, 21:11);
	s(EmcCfgUpdate, 2:0, scratch96, 24:22);
	s(EmcCfgUpdate, 10:8, scratch96, 27:25);
	s(EmcCfgUpdate, 31:28, scratch96, 31:28);
	s(EmcPmacroQuseDdllRank0_1, 10:0, scratch97, 10:0);
	s(EmcPmacroQuseDdllRank0_1, 26:16, scratch97, 21:11);
	s(EmcRfc, 9:0, scratch97, 31:22);
	s(EmcPmacroQuseDdllRank0_2, 10:0, scratch98, 10:0);
	s(EmcPmacroQuseDdllRank0_2, 26:16, scratch98, 21:11);
	s(EmcTxsr, 9:0, scratch98, 31:22);
	s(EmcPmacroQuseDdllRank0_3, 10:0, scratch99, 10:0);
	s(EmcPmacroQuseDdllRank0_3, 26:16, scratch99, 21:11);
	s(EmcMc2EmcQ, 2:0, scratch99, 24:22);
	s(EmcMc2EmcQ, 10:8, scratch99, 27:25);
	s(EmcMc2EmcQ, 27:24, scratch99, 31:28);
	s(EmcPmacroQuseDdllRank0_4, 10:0, scratch100, 10:0);
	s(EmcPmacroQuseDdllRank0_4, 26:16, scratch100, 21:11);
	s(McEmemArbRing1Throttle, 4:0, scratch100, 26:22);
	s(McEmemArbRing1Throttle, 20:16, scratch100, 31:27);
	s(EmcPmacroQuseDdllRank0_5, 10:0, scratch101, 10:0);
	s(EmcPmacroQuseDdllRank0_5, 26:16, scratch101, 21:11);
	s(EmcPmacroQuseDdllRank1_0, 10:0, scratch102, 10:0);
	s(EmcPmacroQuseDdllRank1_0, 26:16, scratch102, 21:11);
	s(EmcAr2Pden, 8:0, scratch102, 30:22);
	s(EmcCfg, 23:23, scratch102, 31:31);
	s(EmcPmacroQuseDdllRank1_1, 10:0, scratch103, 10:0);
	s(EmcPmacroQuseDdllRank1_1, 26:16, scratch103, 21:11);
	s(EmcRfcSlr, 8:0, scratch103, 30:22);
	s(EmcCfg, 24:24, scratch103, 31:31);
	s(EmcPmacroQuseDdllRank1_2, 10:0, scratch104, 10:0);
	s(EmcPmacroQuseDdllRank1_2, 26:16, scratch104, 21:11);
	s(EmcIbdly, 6:0, scratch104, 28:22);
	s(EmcIbdly, 29:28, scratch104, 30:29);
	s(EmcCfg, 25:25, scratch104, 31:31);
	s(EmcPmacroQuseDdllRank1_3, 10:0, scratch105, 10:0);
	s(EmcPmacroQuseDdllRank1_3, 26:16, scratch105, 21:11);
	s(McEmemArbTimingRFCPB, 8:0, scratch105, 30:22);
	s(EmcCfg, 26:26, scratch105, 31:31);
	s(EmcPmacroQuseDdllRank1_4, 10:0, scratch106, 10:0);
	s(EmcPmacroQuseDdllRank1_4, 26:16, scratch106, 21:11);
	s(EmcTfaw, 6:0, scratch106, 28:22);
	s(EmcPmacroDataPadTxCtrl, 3:2, scratch106, 30:29);
	s(EmcCfg, 28:28, scratch106, 31:31);
	s(EmcPmacroQuseDdllRank1_5, 10:0, scratch107, 10:0);
	s(EmcPmacroQuseDdllRank1_5, 26:16, scratch107, 21:11);
	s(EmcTClkStable, 6:0, scratch107, 28:22);
	s(EmcPmacroDataPadTxCtrl, 7:6, scratch107, 30:29);
	s(EmcCfg, 29:29, scratch107, 31:31);
	s(EmcPmacroObDdllLongDqRank0_0, 10:0, scratch108, 10:0);
	s(EmcPmacroObDdllLongDqRank0_0, 26:16, scratch108, 21:11);
	s(EmcPdex2Mrr, 6:0, scratch108, 28:22);
	s(EmcPmacroDataPadTxCtrl, 11:10, scratch108, 30:29);
	s(EmcCfg, 30:30, scratch108, 31:31);
	s(EmcPmacroObDdllLongDqRank0_1, 10:0, scratch109, 10:0);
	s(EmcPmacroObDdllLongDqRank0_1, 26:16, scratch109, 21:11);
	s(EmcRdvMask, 6:0, scratch109, 28:22);
	s(EmcPmacroDataPadTxCtrl, 15:14, scratch109, 30:29);
	s(EmcCfg, 31:31, scratch109, 31:31);
	s(EmcPmacroObDdllLongDqRank0_2, 10:0, scratch110, 10:0);
	s(EmcPmacroObDdllLongDqRank0_2, 26:16, scratch110, 21:11);
	s(EmcRdvEarlyMask, 6:0, scratch110, 28:22);
	s(EmcFbioCfg5, 4:4, scratch110, 29:29);
	s(EmcFbioCfg5, 8:8, scratch110, 30:30);
	s(EmcFbioCfg5, 10:10, scratch110, 31:31);
	s(EmcPmacroObDdllLongDqRank0_3, 10:0, scratch111, 10:0);
	s(EmcPmacroObDdllLongDqRank0_3, 26:16, scratch111, 21:11);
	s(EmcRdvEarly, 6:0, scratch111, 28:22);
	s(EmcFbioCfg5, 12:12, scratch111, 29:29);
	s(EmcFbioCfg5, 25:24, scratch111, 31:30);
	s(EmcPmacroObDdllLongDqRank0_4, 10:0, scratch112, 10:0);
	s(EmcPmacroObDdllLongDqRank0_4, 26:16, scratch112, 21:11);
	s(EmcPmacroDdllShortCmd_2, 6:0, scratch112, 28:22);
	s(EmcFbioCfg5, 28:26, scratch112, 31:29);
	s(EmcPmacroObDdllLongDqRank0_5, 10:0, scratch113, 10:0);
	s(EmcPmacroObDdllLongDqRank0_5, 26:16, scratch113, 21:11);
	s(McEmemArbTimingRp, 6:0, scratch113, 28:22);
	s(EmcFbioCfg5, 31:30, scratch113, 30:29);
	s(EmcCfg2, 0:0, scratch113, 31:31);
	s(EmcPmacroObDdllLongDqRank1_0, 10:0, scratch114, 10:0);
	s(EmcPmacroObDdllLongDqRank1_0, 26:16, scratch114, 21:11);
	s(McEmemArbTimingRas, 6:0, scratch114, 28:22);
	s(EmcCfg2, 2:1, scratch114, 30:29);
	s(EmcCfg2, 7:7, scratch114, 31:31);
	s(EmcPmacroObDdllLongDqRank1_1, 10:0, scratch115, 10:0);
	s(EmcPmacroObDdllLongDqRank1_1, 26:16, scratch115, 21:11);
	s(McEmemArbTimingFaw, 6:0, scratch115, 28:22);
	s(EmcCfg2, 11:10, scratch115, 30:29);
	s(EmcCfg2, 14:14, scratch115, 31:31);
	s(EmcPmacroObDdllLongDqRank1_2, 10:0, scratch123, 10:0);
	s(EmcPmacroObDdllLongDqRank1_2, 26:16, scratch123, 21:11);
	s(McEmemArbTimingRap2Pre, 6:0, scratch123, 28:22);
	s(EmcCfg2, 16:15, scratch123, 30:29);
	s(EmcCfg2, 20:20, scratch123, 31:31);
	s(EmcPmacroObDdllLongDqRank1_3, 10:0, scratch124, 10:0);
	s(EmcPmacroObDdllLongDqRank1_3, 26:16, scratch124, 21:11);
	s(McEmemArbTimingWap2Pre, 6:0, scratch124, 28:22);
	s(EmcCfg2, 24:22, scratch124, 31:29);
	s(EmcPmacroObDdllLongDqRank1_4, 10:0, scratch125, 10:0);
	s(EmcPmacroObDdllLongDqRank1_4, 26:16, scratch125, 21:11);
	s(McEmemArbTimingR2W, 6:0, scratch125, 28:22);
	s(EmcCfg2, 25:25, scratch125, 29:29);
	s(EmcCfg2, 29:28, scratch125, 31:30);
	s(EmcPmacroObDdllLongDqRank1_5, 10:0, scratch126, 10:0);
	s(EmcPmacroObDdllLongDqRank1_5, 26:16, scratch126, 21:11);
	s(McEmemArbTimingW2R, 6:0, scratch126, 28:22);
	s(EmcCfg2, 31:30, scratch126, 30:29);
	s(EmcCfgPipe, 0:0, scratch126, 31:31);
	s(EmcPmacroObDdllLongDqsRank0_0, 10:0, scratch127, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_0, 26:16, scratch127, 21:11);
	s(EmcRp, 5:0, scratch127, 27:22);
	s(EmcCfgPipe, 4:1, scratch127, 31:28);
	s(EmcPmacroObDdllLongDqsRank0_1, 10:0, scratch128, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_1, 26:16, scratch128, 21:11);
	s(EmcR2w, 5:0, scratch128, 27:22);
	s(EmcCfgPipe, 8:5, scratch128, 31:28);
	s(EmcPmacroObDdllLongDqsRank0_2, 10:0, scratch129, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_2, 26:16, scratch129, 21:11);
	s(EmcW2r, 5:0, scratch129, 27:22);
	s(EmcCfgPipe, 11:9, scratch129, 30:28);
	s(EmcCfgPipe, 16:16, scratch129, 31:31);
	s(EmcPmacroObDdllLongDqsRank0_3, 10:0, scratch130, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_3, 26:16, scratch130, 21:11);
	s(EmcR2p, 5:0, scratch130, 27:22);
	s(EmcCfgPipe, 20:17, scratch130, 31:28);
	s(EmcPmacroObDdllLongDqsRank0_4, 10:0, scratch131, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_4, 26:16, scratch131, 21:11);
	s(EmcCcdmw, 5:0, scratch131, 27:22);
	s(EmcCfgPipe, 24:21, scratch131, 31:28);
	s(EmcPmacroObDdllLongDqsRank0_5, 10:0, scratch132, 10:0);
	s(EmcPmacroObDdllLongDqsRank0_5, 26:16, scratch132, 21:11);
	s(EmcRdRcd, 5:0, scratch132, 27:22);
	s(EmcCfgPipe, 27:25, scratch132, 30:28);
	s(EmcPmacroTxPwrd0, 0:0, scratch132, 31:31);
	s(EmcPmacroObDdllLongDqsRank1_0, 10:0, scratch133, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_0, 26:16, scratch133, 21:11);
	s(EmcWrRcd, 5:0, scratch133, 27:22);
	s(EmcPmacroTxPwrd0, 4:1, scratch133, 31:28);
	s(EmcPmacroObDdllLongDqsRank1_1, 10:0, scratch134, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_1, 26:16, scratch134, 21:11);
	s(EmcWdv, 5:0, scratch134, 27:22);
	s(EmcPmacroTxPwrd0, 8:5, scratch134, 31:28);
	s(EmcPmacroObDdllLongDqsRank1_2, 10:0, scratch135, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_2, 26:16, scratch135, 21:11);
	s(EmcQUse, 5:0, scratch135, 27:22);
	s(EmcPmacroTxPwrd0, 12:9, scratch135, 31:28);
	s(EmcPmacroObDdllLongDqsRank1_3, 10:0, scratch136, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_3, 26:16, scratch136, 21:11);
	s(EmcPdEx2Wr, 5:0, scratch136, 27:22);
	s(EmcPmacroTxPwrd0, 13:13, scratch136, 28:28);
	s(EmcPmacroTxPwrd0, 18:16, scratch136, 31:29);
	s(EmcPmacroObDdllLongDqsRank1_4, 10:0, scratch137, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_4, 26:16, scratch137, 21:11);
	s(EmcPdEx2Rd, 5:0, scratch137, 27:22);
	s(EmcPmacroTxPwrd0, 22:19, scratch137, 31:28);
	s(EmcPmacroObDdllLongDqsRank1_5, 10:0, scratch138, 10:0);
	s(EmcPmacroObDdllLongDqsRank1_5, 26:16, scratch138, 21:11);
	s(EmcPdex2Cke, 5:0, scratch138, 27:22);
	s(EmcPmacroTxPwrd0, 26:23, scratch138, 31:28);
	s(EmcPmacroIbDdllLongDqsRank0_0, 10:0, scratch139, 10:0);
	s(EmcPmacroIbDdllLongDqsRank0_0, 26:16, scratch139, 21:11);
	s(EmcPChg2Pden, 5:0, scratch139, 27:22);
	s(EmcPmacroTxPwrd0, 29:27, scratch139, 30:28);
	s(EmcPmacroTxPwrd1, 0:0, scratch139, 31:31);
	s(EmcPmacroIbDdllLongDqsRank0_1, 10:0, scratch140, 10:0);
	s(EmcPmacroIbDdllLongDqsRank0_1, 26:16, scratch140, 21:11);
	s(EmcAct2Pden, 5:0, scratch140, 27:22);
	s(EmcPmacroTxPwrd1, 4:1, scratch140, 31:28);
	s(EmcPmacroIbDdllLongDqsRank0_2, 10:0, scratch141, 10:0);
	s(EmcPmacroIbDdllLongDqsRank0_2, 26:16, scratch141, 21:11);
	s(EmcCke2Pden, 5:0, scratch141, 27:22);
	s(EmcPmacroTxPwrd1, 8:5, scratch141, 31:28);
	s(EmcPmacroIbDdllLongDqsRank0_3, 10:0, scratch142, 10:0);
	s(EmcPmacroIbDdllLongDqsRank0_3, 26:16, scratch142, 21:11);
	s(EmcTcke, 5:0, scratch142, 27:22);
	s(EmcPmacroTxPwrd1, 12:9, scratch142, 31:28);
	s(EmcPmacroIbDdllLongDqsRank1_0, 10:0, scratch143, 10:0);
	s(EmcPmacroIbDdllLongDqsRank1_0, 26:16, scratch143, 21:11);
	s(EmcTrpab, 5:0, scratch143, 27:22);
	s(EmcPmacroTxPwrd1, 13:13, scratch143, 28:28);
	s(EmcPmacroTxPwrd1, 18:16, scratch143, 31:29);
	s(EmcPmacroIbDdllLongDqsRank1_1, 10:0, scratch144, 10:0);
	s(EmcPmacroIbDdllLongDqsRank1_1, 26:16, scratch144, 21:11);
	s(EmcClkenOverride, 3:1, scratch144, 24:22);
	s(EmcClkenOverride, 8:6, scratch144, 27:25);
	s(EmcPmacroTxPwrd1, 22:19, scratch144, 31:28);
	s(EmcPmacroIbDdllLongDqsRank1_2, 10:0, scratch145, 10:0);
	s(EmcPmacroIbDdllLongDqsRank1_2, 26:16, scratch145, 21:11);
	s(EmcEInput, 5:0, scratch145, 27:22);
	s(EmcPmacroTxPwrd1, 26:23, scratch145, 31:28);
	s(EmcPmacroIbDdllLongDqsRank1_3, 10:0, scratch146, 10:0);
	s(EmcPmacroIbDdllLongDqsRank1_3, 26:16, scratch146, 21:11);
	s(EmcEInputDuration, 5:0, scratch146, 27:22);
	s(EmcPmacroTxPwrd1, 29:27, scratch146, 30:28);
	s(EmcPmacroTxPwrd2, 0:0, scratch146, 31:31);
	s(EmcPmacroDdllLongCmd_0, 10:0, scratch147, 10:0);
	s(EmcPmacroDdllLongCmd_0, 26:16, scratch147, 21:11);
	s(EmcPutermExtra, 5:0, scratch147, 27:22);
	s(EmcPmacroTxPwrd2, 4:1, scratch147, 31:28);
	s(EmcPmacroDdllLongCmd_1, 10:0, scratch148, 10:0);
	s(EmcPmacroDdllLongCmd_1, 26:16, scratch148, 21:11);
	s(EmcTckesr, 5:0, scratch148, 27:22);
	s(EmcPmacroTxPwrd2, 8:5, scratch148, 31:28);
	s(EmcPmacroDdllLongCmd_2, 10:0, scratch149, 10:0);
	s(EmcPmacroDdllLongCmd_2, 26:16, scratch149, 21:11);
	s(EmcTpd, 5:0, scratch149, 27:22);
	s(EmcPmacroTxPwrd2, 12:9, scratch149, 31:28);
	s(EmcPmacroDdllLongCmd_3, 10:0, scratch150, 10:0);
	s(EmcPmacroDdllLongCmd_3, 26:16, scratch150, 21:11);
	s(EmcWdvMask, 5:0, scratch150, 27:22);
	s(EmcPmacroTxPwrd2, 13:13, scratch150, 28:28);
	s(EmcPmacroTxPwrd2, 18:16, scratch150, 31:29);
	s(McEmemArbCfg, 8:0, scratch151, 8:0);
	s(McEmemArbCfg, 20:16, scratch151, 13:9);
	s(McEmemArbCfg, 31:24, scratch151, 21:14);
	s(EmcWdvChk, 5:0, scratch151, 27:22);
	s(EmcPmacroTxPwrd2, 22:19, scratch151, 31:28);
	s(McEmemArbMisc1, 12:0, scratch152, 12:0);
	s(McEmemArbMisc1, 25:21, scratch152, 17:13);
	s(McEmemArbMisc1, 31:28, scratch152, 21:18);
	s(EmcCmdBrlshft0, 5:0, scratch152, 27:22);
	s(EmcPmacroTxPwrd2, 26:23, scratch152, 31:28);
	s(EmcMrsWaitCnt2, 9:0, scratch153, 9:0);
	s(EmcMrsWaitCnt2, 26:16, scratch153, 20:10);
	s(EmcPmacroIbRxrt, 10:0, scratch153, 31:21);
	s(EmcMrsWaitCnt, 9:0, scratch154, 9:0);
	s(EmcMrsWaitCnt, 26:16, scratch154, 20:10);
	s(EmcPmacroDdllLongCmd_4, 10:0, scratch154, 31:21);
	s(EmcAutoCalInterval, 20:0, scratch155, 20:0);
	s(McEmemArbOutstandingReq, 8:0, scratch155, 29:21);
	s(McEmemArbOutstandingReq, 31:30, scratch155, 31:30);
	s(McEmemArbRefpbHpCtrl, 6:0, scratch156, 6:0);
	s(McEmemArbRefpbHpCtrl, 14:8, scratch156, 13:7);
	s(McEmemArbRefpbHpCtrl, 22:16, scratch156, 20:14);
	s(EmcCmdBrlshft1, 5:0, scratch156, 26:21);
	s(EmcRrd, 4:0, scratch156, 31:27);
	s(EmcQuseBrlshft0, 19:0, scratch157, 19:0);
	s(EmcFbioCfg8, 27:16, scratch157, 31:20);
	s(EmcQuseBrlshft1, 19:0, scratch158, 19:0);
	s(EmcTxsrDll, 11:0, scratch158, 31:20);
	s(EmcQuseBrlshft2, 19:0, scratch159, 19:0);
	s(EmcTxdsrvttgen, 11:0, scratch159, 31:20);
	s(EmcQuseBrlshft3, 19:0, scratch160, 19:0);
	s(EmcPmacroVttgenCtrl0, 3:0, scratch160, 23:20);
	s(EmcPmacroVttgenCtrl0, 11:8, scratch160, 27:24);
	s(EmcPmacroVttgenCtrl0, 19:16, scratch160, 31:28);
	s(EmcPmacroVttgenCtrl1, 19:0, scratch161, 19:0);
	s(EmcCmdBrlshft2, 5:0, scratch161, 25:20);
	s(EmcCmdBrlshft3, 5:0, scratch161, 31:26);
	s(EmcAutoCalConfig3, 5:0, scratch162, 5:0);
	s(EmcAutoCalConfig3, 13:8, scratch162, 11:6);
	s(EmcAutoCalConfig3, 18:16, scratch162, 14:12);
	s(EmcAutoCalConfig3, 22:20, scratch162, 17:15);
	s(EmcTRefBw, 13:0, scratch162, 31:18);
	s(EmcAutoCalConfig4, 5:0, scratch163, 5:0);
	s(EmcAutoCalConfig4, 13:8, scratch163, 11:6);
	s(EmcAutoCalConfig4, 18:16, scratch163, 14:12);
	s(EmcAutoCalConfig4, 22:20, scratch163, 17:15);
	s(EmcQpop, 6:0, scratch163, 24:18);
	s(EmcQpop, 22:16, scratch163, 31:25);
	s(EmcAutoCalConfig5, 5:0, scratch164, 5:0);
	s(EmcAutoCalConfig5, 13:8, scratch164, 11:6);
	s(EmcAutoCalConfig5, 18:16, scratch164, 14:12);
	s(EmcAutoCalConfig5, 22:20, scratch164, 17:15);
	s(EmcPmacroAutocalCfgCommon, 5:0, scratch164, 23:18);
	s(EmcPmacroAutocalCfgCommon, 13:8, scratch164, 29:24);
	s(EmcPmacroAutocalCfgCommon, 16:16, scratch164, 30:30);
	s(EmcPmacroTxPwrd2, 27:27, scratch164, 31:31);
	s(EmcAutoCalConfig6, 5:0, scratch165, 5:0);
	s(EmcAutoCalConfig6, 13:8, scratch165, 11:6);
	s(EmcAutoCalConfig6, 18:16, scratch165, 14:12);
	s(EmcAutoCalConfig6, 22:20, scratch165, 17:15);
	s(EmcWev, 5:0, scratch165, 23:18);
	s(EmcWsv, 5:0, scratch165, 29:24);
	s(EmcPmacroTxPwrd2, 29:28, scratch165, 31:30);
	s(EmcAutoCalConfig7, 5:0, scratch166, 5:0);
	s(EmcAutoCalConfig7, 13:8, scratch166, 11:6);
	s(EmcAutoCalConfig7, 18:16, scratch166, 14:12);
	s(EmcAutoCalConfig7, 22:20, scratch166, 17:15);
	s(EmcCfg3, 2:0, scratch166, 20:18);
	s(EmcCfg3, 6:4, scratch166, 23:21);
	s(EmcQuseWidth, 3:0, scratch166, 27:24);
	s(EmcQuseWidth, 29:28, scratch166, 29:28);
	s(EmcPmacroTxPwrd3, 1:0, scratch166, 31:30);
	s(EmcAutoCalConfig8, 5:0, scratch167, 5:0);
	s(EmcAutoCalConfig8, 13:8, scratch167, 11:6);
	s(EmcAutoCalConfig8, 18:16, scratch167, 14:12);
	s(EmcAutoCalConfig8, 22:20, scratch167, 17:15);
	s(EmcPmacroBgBiasCtrl0, 2:0, scratch167, 20:18);
	s(EmcPmacroBgBiasCtrl0, 6:4, scratch167, 23:21);
	s(McEmemArbTimingRcd, 5:0, scratch167, 29:24);
	s(EmcPmacroTxPwrd3, 3:2, scratch167, 31:30);
	s(EmcXm2CompPadCtrl2, 17:0, scratch168, 17:0);
	s(McEmemArbTimingCcdmw, 5:0, scratch168, 23:18);
	s(McEmemArbOverride, 27:27, scratch168, 24:24);
	s(McEmemArbOverride, 26:26, scratch168, 25:25);
	s(McEmemArbOverride, 16:16, scratch168, 26:26);
	s(McEmemArbOverride, 10:10, scratch168, 27:27);
	s(McEmemArbOverride, 4:4, scratch168, 28:28);
	s(McEmemArbOverride, 3:3, scratch168, 29:29);
	s(EmcPmacroTxPwrd3, 5:4, scratch168, 31:30);
	s(EmcXm2CompPadCtrl3, 17:0, scratch169, 17:0);
	s(EmcRext, 4:0, scratch169, 22:18);
	s(EmcTClkStop, 4:0, scratch169, 27:23);
	s(EmcPmacroTxPwrd3, 9:6, scratch169, 31:28);
	s(EmcZcalWaitCnt, 10:0, scratch170, 10:0);
	s(EmcZcalWaitCnt, 21:16, scratch170, 16:11);
	s(EmcZcalWaitCnt, 31:31, scratch170, 17:17);
	s(EmcWext, 4:0, scratch170, 22:18);
	s(EmcRefctrl2, 0:0, scratch170, 23:23);
	s(EmcRefctrl2, 26:24, scratch170, 26:24);
	s(EmcRefctrl2, 31:31, scratch170, 27:27);
	s(EmcPmacroTxPwrd3, 13:10, scratch170, 31:28);
	s(EmcZcalMrwCmd, 7:0, scratch171, 7:0);
	s(EmcZcalMrwCmd, 23:16, scratch171, 15:8);
	s(EmcZcalMrwCmd, 31:30, scratch171, 17:16);
	s(EmcWeDuration, 4:0, scratch171, 22:18);
	s(EmcWsDuration, 4:0, scratch171, 27:23);
	s(EmcPmacroTxPwrd3, 19:16, scratch171, 31:28);
	s(EmcSwizzleRank0Byte0, 2:0, scratch172, 2:0);
	s(EmcSwizzleRank0Byte0, 6:4, scratch172, 5:3);
	s(EmcSwizzleRank0Byte0, 10:8, scratch172, 8:6);
	s(EmcSwizzleRank0Byte0, 14:12, scratch172, 11:9);
	s(EmcSwizzleRank0Byte0, 18:16, scratch172, 14:12);
	s(EmcSwizzleRank0Byte0, 22:20, scratch172, 17:15);
	s(EmcPutermWidth, 31:31, scratch172, 18:18);
	s(EmcPutermWidth, 3:0, scratch172, 22:19);
	s(McEmemArbTimingRrd, 4:0, scratch172, 27:23);
	s(EmcPmacroTxPwrd3, 23:20, scratch172, 31:28);
	s(EmcSwizzleRank0Byte1, 2:0, scratch173, 2:0);
	s(EmcSwizzleRank0Byte1, 6:4, scratch173, 5:3);
	s(EmcSwizzleRank0Byte1, 10:8, scratch173, 8:6);
	s(EmcSwizzleRank0Byte1, 14:12, scratch173, 11:9);
	s(EmcSwizzleRank0Byte1, 18:16, scratch173, 14:12);
	s(EmcSwizzleRank0Byte1, 22:20, scratch173, 17:15);
	s(McEmemArbTimingR2R, 4:0, scratch173, 22:18);
	s(McEmemArbTimingW2W, 4:0, scratch173, 27:23);
	s(EmcPmacroTxPwrd3, 27:24, scratch173, 31:28);
	s(EmcSwizzleRank0Byte2, 2:0, scratch174, 2:0);
	s(EmcSwizzleRank0Byte2, 6:4, scratch174, 5:3);
	s(EmcSwizzleRank0Byte2, 10:8, scratch174, 8:6);
	s(EmcSwizzleRank0Byte2, 14:12, scratch174, 11:9);
	s(EmcSwizzleRank0Byte2, 18:16, scratch174, 14:12);
	s(EmcSwizzleRank0Byte2, 22:20, scratch174, 17:15);
	s(EmcPmacroTxPwrd3, 29:28, scratch174, 19:18);
	s(EmcPmacroTxSelClkSrc0, 11:0, scratch174, 31:20);
	s(EmcSwizzleRank0Byte3, 2:0, scratch175, 2:0);
	s(EmcSwizzleRank0Byte3, 6:4, scratch175, 5:3);
	s(EmcSwizzleRank0Byte3, 10:8, scratch175, 8:6);
	s(EmcSwizzleRank0Byte3, 14:12, scratch175, 11:9);
	s(EmcSwizzleRank0Byte3, 18:16, scratch175, 14:12);
	s(EmcSwizzleRank0Byte3, 22:20, scratch175, 17:15);
	s(EmcPmacroTxSelClkSrc0, 27:16, scratch175, 29:18);
	s(EmcPmacroTxSelClkSrc1, 1:0, scratch175, 31:30);
	s(EmcSwizzleRank1Byte0, 2:0, scratch176, 2:0);
	s(EmcSwizzleRank1Byte0, 6:4, scratch176, 5:3);
	s(EmcSwizzleRank1Byte0, 10:8, scratch176, 8:6);
	s(EmcSwizzleRank1Byte0, 14:12, scratch176, 11:9);
	s(EmcSwizzleRank1Byte0, 18:16, scratch176, 14:12);
	s(EmcSwizzleRank1Byte0, 22:20, scratch176, 17:15);
	s(EmcPmacroTxSelClkSrc1, 11:2, scratch176, 27:18);
	s(EmcPmacroTxSelClkSrc1, 19:16, scratch176, 31:28);
	s(EmcSwizzleRank1Byte1, 2:0, scratch177, 2:0);
	s(EmcSwizzleRank1Byte1, 6:4, scratch177, 5:3);
	s(EmcSwizzleRank1Byte1, 10:8, scratch177, 8:6);
	s(EmcSwizzleRank1Byte1, 14:12, scratch177, 11:9);
	s(EmcSwizzleRank1Byte1, 18:16, scratch177, 14:12);
	s(EmcSwizzleRank1Byte1, 22:20, scratch177, 17:15);
	s(EmcPmacroTxSelClkSrc1, 27:20, scratch177, 25:18);
	s(EmcPmacroTxSelClkSrc3, 5:0, scratch177, 31:26);
	s(EmcSwizzleRank1Byte2, 2:0, scratch178, 2:0);
	s(EmcSwizzleRank1Byte2, 6:4, scratch178, 5:3);
	s(EmcSwizzleRank1Byte2, 10:8, scratch178, 8:6);
	s(EmcSwizzleRank1Byte2, 14:12, scratch178, 11:9);
	s(EmcSwizzleRank1Byte2, 18:16, scratch178, 14:12);
	s(EmcSwizzleRank1Byte2, 22:20, scratch178, 17:15);
	s(EmcPmacroTxSelClkSrc3, 11:6, scratch178, 23:18);
	s(EmcPmacroTxSelClkSrc3, 23:16, scratch178, 31:24);
	s(EmcSwizzleRank1Byte3, 2:0, scratch179, 2:0);
	s(EmcSwizzleRank1Byte3, 6:4, scratch179, 5:3);
	s(EmcSwizzleRank1Byte3, 10:8, scratch179, 8:6);
	s(EmcSwizzleRank1Byte3, 14:12, scratch179, 11:9);
	s(EmcSwizzleRank1Byte3, 18:16, scratch179, 14:12);
	s(EmcSwizzleRank1Byte3, 22:20, scratch179, 17:15);
	s(EmcPmacroTxSelClkSrc3, 27:24, scratch179, 21:18);
	s(EmcPmacroTxSelClkSrc2, 9:0, scratch179, 31:22);
	s(EmcPmacroCmdBrickCtrlFdpd, 17:0, scratch180, 17:0);
	s(EmcPmacroTxSelClkSrc2, 11:10, scratch180, 19:18);
	s(EmcPmacroTxSelClkSrc2, 27:16, scratch180, 31:20);
	s(EmcPmacroDataBrickCtrlFdpd, 17:0, scratch181, 17:0);
	s(EmcPmacroTxSelClkSrc4, 11:0, scratch181, 29:18);
	s(EmcPmacroTxSelClkSrc4, 17:16, scratch181, 31:30);
	s(EmcFbioCfg7, 16:0, scratch182, 16:0);
	s(McEmemArbRefpbBankCtrl, 6:0, scratch182, 23:17);
	s(McEmemArbRefpbBankCtrl, 14:8, scratch182, 30:24);
	s(McEmemArbRefpbBankCtrl, 31:31, scratch182, 31:31);
	s(EmcDynSelfRefControl, 15:0, scratch183, 15:0);
	s(EmcDynSelfRefControl, 31:31, scratch183, 16:16);
	s(EmcPmacroTxSelClkSrc4, 27:18, scratch183, 26:17);
	s(EmcPmacroTxSelClkSrc5, 4:0, scratch183, 31:27);
	s(EmcDllCfg1, 16:0, scratch184, 16:0);
	s(EmcPmacroTxSelClkSrc5, 11:5, scratch184, 23:17);
	s(EmcPmacroTxSelClkSrc5, 23:16, scratch184, 31:24);
	s(EmcPmacroPadCfgCtrl, 1:0, scratch185, 1:0);
	s(EmcPmacroPadCfgCtrl, 6:5, scratch185, 3:2);
	s(EmcPmacroPadCfgCtrl, 11:9, scratch185, 6:4);
	s(EmcPmacroPadCfgCtrl, 13:13, scratch185, 7:7);
	s(EmcPmacroPadCfgCtrl, 17:16, scratch185, 9:8);
	s(EmcPmacroPadCfgCtrl, 21:20, scratch185, 11:10);
	s(EmcPmacroPadCfgCtrl, 25:24, scratch185, 13:12);
	s(EmcPmacroPadCfgCtrl, 30:28, scratch185, 16:14);
	s(EmcPmacroTxSelClkSrc5, 27:24, scratch185, 20:17);
	s(EmcPmacroCmdPadTxCtrl, 1:0, scratch185, 22:21);
	s(EmcPmacroCmdPadTxCtrl, 5:4, scratch185, 24:23);
	s(EmcPmacroCmdPadTxCtrl, 9:8, scratch185, 26:25);
	s(EmcPmacroCmdPadTxCtrl, 13:12, scratch185, 28:27);
	s(EmcPmacroCmdPadTxCtrl, 16:16, scratch185, 29:29);
	s(EmcPmacroCmdPadTxCtrl, 21:20, scratch185, 31:30);
	s(EmcRefresh, 15:0, scratch186, 15:0);
	s(EmcCmdQ, 4:0, scratch186, 20:16);
	s(EmcCmdQ, 10:8, scratch186, 23:21);
	s(EmcCmdQ, 14:12, scratch186, 26:24);
	s(EmcCmdQ, 28:24, scratch186, 31:27);
	s(EmcAcpdControl, 15:0, scratch187, 15:0);
	s(EmcAutoCalVrefSel1, 15:0, scratch187, 31:16);
	s(EmcXm2CompPadCtrl, 1:0, scratch188, 1:0);
	s(EmcXm2CompPadCtrl, 6:3, scratch188, 5:2);
	s(EmcXm2CompPadCtrl, 9:9, scratch188, 6:6);
	s(EmcXm2CompPadCtrl, 19:11, scratch188, 15:7);
	s(EmcCfgDigDllPeriod, 15:0, scratch188, 31:16);
	s(EmcCfgDigDll_1, 15:0, scratch189, 15:0);
	s(EmcPreRefreshReqCnt, 15:0, scratch189, 31:16);
	s(EmcPmacroCmdPadTxCtrl, 27:24, scratch190, 19:16);
	s(EmcPmacroDataPadTxCtrl, 1:0, scratch190, 21:20);
	s(EmcPmacroDataPadTxCtrl, 5:4, scratch190, 23:22);
	s(EmcPmacroDataPadTxCtrl, 9:8, scratch190, 25:24);
	s(EmcPmacroDataPadTxCtrl, 13:12, scratch190, 27:26);
	s(EmcPmacroDataPadTxCtrl, 16:16, scratch190, 28:28);
	s(EmcPmacroDataPadTxCtrl, 21:20, scratch190, 30:29);
	s(EmcPmacroDataPadTxCtrl, 24:24, scratch190, 31:31);
	s(EmcPmacroDataPadTxCtrl, 27:25, scratch191, 2:0);

	s(EmcPinGpio, 1:0, scratch8, 31:30);
	s(EmcPinGpioEn, 1:0, scratch9, 31:30);
	s(EmcDevSelect, 1:0, scratch10, 31:30);
	s(EmcZcalWarmColdBootEnables, 1:0, scratch11, 31:30);
	s(EmcCfgDigDllPeriodWarmBoot, 1:0, scratch12, 31:30);
	s32(EmcBctSpare13, scratch31);
	s32(EmcBctSpare12, scratch32);
	s32(EmcBctSpare7, scratch33);
	s32(EmcBctSpare6, scratch40);
	s32(EmcBctSpare5, scratch42);
	s32(EmcBctSpare4, scratch44);
	s32(SwizzleRankByteEncode, scratch45);
	s32(EmcBctSpare2, scratch46);
	s32(EmcBctSpare1, scratch47);
	s32(EmcBctSpare0, scratch48);
	s32(EmcBctSpare9, scratch50);
	s32(EmcBctSpare8, scratch51);
	s32(BootRomPatchData, scratch56);
	s32(BootRomPatchControl, scratch57);
	s(McClkenOverrideAllWarmBoot, 0:0, scratch58, 31:31);
	s(EmcClkenOverrideAllWarmBoot, 0:0, scratch59, 30:30);
	s(EmcMrsWarmBootEnable, 0:0, scratch59, 31:31);
	s(ClearClk2Mc1, 0:0, scratch60, 30:30);
	s(EmcWarmBootExtraModeRegWriteEnable, 0:0, scratch60, 31:31);
	s(ClkRstControllerPllmMisc2OverrideEnable, 0:0, scratch61, 30:30);
	s(EmcDbgWriteMux, 0:0, scratch61, 31:31);
	s(EmcExtraRefreshNum, 2:0, scratch62, 31:29);
	s(PmcIoDpd3ReqWait, 2:0, scratch68, 30:28);
	s(AhbArbitrationXbarCtrlMemInitDone, 0:0, scratch68, 31:31);
	s(MemoryType, 2:0, scratch69, 30:28);
	s(PmcIoDpd4ReqWait, 2:0, scratch70, 30:28);
	s(EmcTimingControlWait, 7:0, scratch86, 31:24);
	s(EmcZcalWarmBootWait, 7:0, scratch87, 31:24);
	s(WarmBootWait, 7:0, scratch88, 31:24);
	s(EmcPinProgramWait, 7:0, scratch89, 31:24);
	s(EmcAutoCalWait, 9:0, scratch101, 31:22);
	s(SwizzleRankByteEncode, 15:0, scratch190, 15:0);

	switch (sdram->MemoryType) {
	case NvBootMemoryType_LpDdr2:
	case NvBootMemoryType_LpDdr4:
		s(EmcMrwLpddr2ZcalWarmBoot, 23:16, scratch5, 7:0);
		s(EmcMrwLpddr2ZcalWarmBoot, 7:0, scratch5, 15:8);
		s(EmcWarmBootMrwExtra, 23:16, scratch5, 23:16);
		s(EmcWarmBootMrwExtra, 7:0, scratch5, 31:24);
		s(EmcMrwLpddr2ZcalWarmBoot, 31:30, scratch6, 1:0);
		s(EmcWarmBootMrwExtra, 31:30, scratch6, 3:2);
		s(EmcMrwLpddr2ZcalWarmBoot, 27:26, scratch6, 5:4);
		s(EmcWarmBootMrwExtra, 27:26, scratch6, 7:6);
		s(EmcMrw6, 27:0, scratch8, 27:0);
		s(EmcMrw6, 31:30, scratch8, 29:28);
		s(EmcMrw8, 27:0, scratch9, 27:0);
		s(EmcMrw8, 31:30, scratch9, 29:28);
		s(EmcMrw9, 27:0, scratch10, 27:0);
		s(EmcMrw9, 31:30, scratch10, 29:28);
		s(EmcMrw10, 27:0, scratch11, 27:0);
		s(EmcMrw10, 31:30, scratch11, 29:28);
		s(EmcMrw12, 27:0, scratch12, 27:0);
		s(EmcMrw12, 31:30, scratch12, 29:28);
		s(EmcMrw13, 27:0, scratch13, 27:0);
		s(EmcMrw13, 31:30, scratch13, 29:28);
		s(EmcMrw14, 27:0, scratch14, 27:0);
		s(EmcMrw14, 31:30, scratch14, 29:28);
		s(EmcMrw1, 7:0, scratch15, 7:0);
		s(EmcMrw1, 23:16, scratch15, 15:8);
		s(EmcMrw1, 27:26, scratch15, 17:16);
		s(EmcMrw1, 31:30, scratch15, 19:18);
		s(EmcWarmBootMrwExtra, 7:0, scratch16, 7:0);
		s(EmcWarmBootMrwExtra, 23:16, scratch16, 15:8);
		s(EmcWarmBootMrwExtra, 27:26, scratch16, 17:16);
		s(EmcWarmBootMrwExtra, 31:30, scratch16, 19:18);
		s(EmcMrw2, 7:0, scratch17, 7:0);
		s(EmcMrw2, 23:16, scratch17, 15:8);
		s(EmcMrw2, 27:26, scratch17, 17:16);
		s(EmcMrw2, 31:30, scratch17, 19:18);
		s(EmcMrw3, 7:0, scratch18, 7:0);
		s(EmcMrw3, 23:16, scratch18, 15:8);
		s(EmcMrw3, 27:26, scratch18, 17:16);
		s(EmcMrw3, 31:30, scratch18, 19:18);
		s(EmcMrw4, 7:0, scratch19, 7:0);
		s(EmcMrw4, 23:16, scratch19, 15:8);
		s(EmcMrw4, 27:26, scratch19, 17:16);
		s(EmcMrw4, 31:30, scratch19, 19:18);
		break;
	case NvBootMemoryType_Ddr3:
		s(EmcMrs, 13:0, scratch5, 13:0);
		s(EmcEmrs, 13:0, scratch5, 27:14);
		s(EmcMrs, 21:20, scratch5, 29:28);
		s(EmcMrs, 31:30, scratch5, 31:30);
		s(EmcEmrs2, 13:0, scratch8, 13:0);
		s(EmcEmrs3, 13:0, scratch8, 27:14);
		s(EmcEmrs, 21:20, scratch8, 29:28);
		s(EmcWarmBootMrsExtra, 13:0, scratch9, 13:0);
		s(EmcEmrs, 31:30, scratch9, 15:14);
		s(EmcEmrs2, 21:20, scratch9, 17:16);
		s(EmcEmrs2, 31:30, scratch9, 19:18);
		s(EmcEmrs3, 21:20, scratch9, 21:20);
		s(EmcEmrs3, 31:30, scratch9, 23:22);
		s(EmcWarmBootMrsExtra, 31:30, scratch9, 25:24);
		s(EmcWarmBootMrsExtra, 21:20, scratch9, 27:26);
		s(EmcZqCalDdr3WarmBoot, 31:30, scratch9, 29:28);
		s(EmcMrs, 27:26, scratch10, 1:0);
		s(EmcEmrs, 27:26, scratch10, 3:2);
		s(EmcEmrs2, 27:26, scratch10, 5:4);
		s(EmcEmrs3, 27:26, scratch10, 7:6);
		s(EmcWarmBootMrsExtra, 27:27, scratch10, 8:8);
		s(EmcWarmBootMrsExtra, 26:26, scratch10, 9:9);
		s(EmcZqCalDdr3WarmBoot, 0:0, scratch10, 10:10);
		s(EmcZqCalDdr3WarmBoot, 4:4, scratch10, 11:11);
		break;
	default:
		printk(BIOS_CRIT, "ERROR: %s() unrecognized MemoryType %d!\n",
		       __func__, sdram->MemoryType);
	}

	s32(EmcCmdMappingByte, secure_scratch8);
	s32(EmcPmacroBrickMapping0, secure_scratch9);
	s32(EmcPmacroBrickMapping1, secure_scratch10);
	s32(EmcPmacroBrickMapping2, secure_scratch11);
	s32(McVideoProtectGpuOverride0, secure_scratch12);
	s(EmcCmdMappingCmd0_0, 6:0, secure_scratch13, 6:0);
	s(EmcCmdMappingCmd0_0, 14:8, secure_scratch13, 13:7);
	s(EmcCmdMappingCmd0_0, 22:16, secure_scratch13, 20:14);
	s(EmcCmdMappingCmd0_0, 30:24, secure_scratch13, 27:21);
	s(McVideoProtectBomAdrHi, 1:0, secure_scratch13, 29:28);
	s(McVideoProtectWriteAccess, 1:0, secure_scratch13, 31:30);
	s(EmcCmdMappingCmd0_1, 6:0, secure_scratch14, 6:0);
	s(EmcCmdMappingCmd0_1, 14:8, secure_scratch14, 13:7);
	s(EmcCmdMappingCmd0_1, 22:16, secure_scratch14, 20:14);
	s(EmcCmdMappingCmd0_1, 30:24, secure_scratch14, 27:21);
	s(McSecCarveoutAdrHi, 1:0, secure_scratch14, 29:28);
	s(McMtsCarveoutAdrHi, 1:0, secure_scratch14, 31:30);
	s(EmcCmdMappingCmd1_0, 6:0, secure_scratch15, 6:0);
	s(EmcCmdMappingCmd1_0, 14:8, secure_scratch15, 13:7);
	s(EmcCmdMappingCmd1_0, 22:16, secure_scratch15, 20:14);
	s(EmcCmdMappingCmd1_0, 30:24, secure_scratch15, 27:21);
	s(McGeneralizedCarveout5BomHi, 1:0, secure_scratch15, 29:28);
	s(McGeneralizedCarveout3BomHi, 1:0, secure_scratch15, 31:30);
	s(EmcCmdMappingCmd1_1, 6:0, secure_scratch16, 6:0);
	s(EmcCmdMappingCmd1_1, 14:8, secure_scratch16, 13:7);
	s(EmcCmdMappingCmd1_1, 22:16, secure_scratch16, 20:14);
	s(EmcCmdMappingCmd1_1, 30:24, secure_scratch16, 27:21);
	s(McGeneralizedCarveout2BomHi, 1:0, secure_scratch16, 29:28);
	s(McGeneralizedCarveout4BomHi, 1:0, secure_scratch16, 31:30);
	s(EmcCmdMappingCmd2_0, 6:0, secure_scratch17, 6:0);
	s(EmcCmdMappingCmd2_0, 14:8, secure_scratch17, 13:7);
	s(EmcCmdMappingCmd2_0, 22:16, secure_scratch17, 20:14);
	s(EmcCmdMappingCmd2_0, 30:24, secure_scratch17, 27:21);
	s(McGeneralizedCarveout1BomHi, 1:0, secure_scratch17, 29:28);
	s(EmcAdrCfg, 0:0, secure_scratch17, 30:30);
	s(EmcFbioSpare, 1:1, secure_scratch17, 31:31);
	s(EmcCmdMappingCmd2_1, 6:0, secure_scratch18, 6:0);
	s(EmcCmdMappingCmd2_1, 14:8, secure_scratch18, 13:7);
	s(EmcCmdMappingCmd2_1, 22:16, secure_scratch18, 20:14);
	s(EmcCmdMappingCmd2_1, 30:24, secure_scratch18, 27:21);
	s(EmcFbioCfg8, 15:15, secure_scratch18, 28:28);
	s(McEmemAdrCfg, 0:0, secure_scratch18, 29:29);
	s(McSecCarveoutProtectWriteAccess, 0:0, secure_scratch18, 30:30);
	s(McMtsCarveoutRegCtrl, 0:0, secure_scratch18, 31:31);
	s(EmcCmdMappingCmd3_0, 6:0, secure_scratch19, 6:0);
	s(EmcCmdMappingCmd3_0, 14:8, secure_scratch19, 13:7);
	s(EmcCmdMappingCmd3_0, 22:16, secure_scratch19, 20:14);
	s(EmcCmdMappingCmd3_0, 30:24, secure_scratch19, 27:21);
	s(McGeneralizedCarveout2Cfg0, 6:3, secure_scratch19, 31:28);
	s(EmcCmdMappingCmd3_1, 6:0, secure_scratch20, 6:0);
	s(EmcCmdMappingCmd3_1, 14:8, secure_scratch20, 13:7);
	s(EmcCmdMappingCmd3_1, 22:16, secure_scratch20, 20:14);
	s(EmcCmdMappingCmd3_1, 30:24, secure_scratch20, 27:21);
	s(McGeneralizedCarveout2Cfg0, 10:7, secure_scratch20, 31:28);
	s(McGeneralizedCarveout4Cfg0, 26:0, secure_scratch39, 26:0);
	s(McGeneralizedCarveout2Cfg0, 17:14, secure_scratch39, 30:27);
	s(McVideoProtectVprOverride, 0:0, secure_scratch39, 31:31);
	s(McGeneralizedCarveout5Cfg0, 26:0, secure_scratch40, 26:0);
	s(McGeneralizedCarveout2Cfg0, 21:18, secure_scratch40, 30:27);
	s(McVideoProtectVprOverride, 1:1, secure_scratch40, 31:31);
	s(EmcCmdMappingCmd0_2, 6:0, secure_scratch41, 6:0);
	s(EmcCmdMappingCmd0_2, 14:8, secure_scratch41, 13:7);
	s(EmcCmdMappingCmd0_2, 22:16, secure_scratch41, 20:14);
	s(EmcCmdMappingCmd0_2, 27:24, secure_scratch41, 24:21);
	s(McGeneralizedCarveout1Cfg0, 6:3, secure_scratch41, 28:25);
	s(McGeneralizedCarveout2Cfg0, 13:11, secure_scratch41, 31:29);
	s(EmcCmdMappingCmd1_2, 6:0, secure_scratch42, 6:0);
	s(EmcCmdMappingCmd1_2, 14:8, secure_scratch42, 13:7);
	s(EmcCmdMappingCmd1_2, 22:16, secure_scratch42, 20:14);
	s(EmcCmdMappingCmd1_2, 27:24, secure_scratch42, 24:21);
	s(McGeneralizedCarveout1Cfg0, 13:7, secure_scratch42, 31:25);
	s(EmcCmdMappingCmd2_2, 6:0, secure_scratch43, 6:0);
	s(EmcCmdMappingCmd2_2, 14:8, secure_scratch43, 13:7);
	s(EmcCmdMappingCmd2_2, 22:16, secure_scratch43, 20:14);
	s(EmcCmdMappingCmd2_2, 27:24, secure_scratch43, 24:21);
	s(McGeneralizedCarveout1Cfg0, 17:14, secure_scratch43, 28:25);
	s(McGeneralizedCarveout3Cfg0, 13:11, secure_scratch43, 31:29);
	s(EmcCmdMappingCmd3_2, 6:0, secure_scratch44, 6:0);
	s(EmcCmdMappingCmd3_2, 14:8, secure_scratch44, 13:7);
	s(EmcCmdMappingCmd3_2, 22:16, secure_scratch44, 20:14);
	s(EmcCmdMappingCmd3_2, 27:24, secure_scratch44, 24:21);
	s(McGeneralizedCarveout1Cfg0, 21:18, secure_scratch44, 28:25);
	s(McVideoProtectVprOverride, 3:2, secure_scratch44, 30:29);
	s(McVideoProtectVprOverride, 6:6, secure_scratch44, 31:31);
	s(McEmemAdrCfgChannelMask, 31:9, secure_scratch45, 22:0);
	s(McEmemAdrCfgDev0, 2:0, secure_scratch45, 25:23);
	s(McEmemAdrCfgDev0, 9:8, secure_scratch45, 27:26);
	s(McEmemAdrCfgDev0, 19:16, secure_scratch45, 31:28);
	s(McEmemAdrCfgBankMask0, 31:10, secure_scratch46, 21:0);
	s(McEmemAdrCfgDev1, 2:0, secure_scratch46, 24:22);
	s(McEmemAdrCfgDev1, 9:8, secure_scratch46, 26:25);
	s(McEmemAdrCfgDev1, 19:16, secure_scratch46, 30:27);
	s(McVideoProtectVprOverride, 7:7, secure_scratch46, 31:31);
	s(McEmemAdrCfgBankMask1, 31:10, secure_scratch47, 21:0);
	s(McGeneralizedCarveout3Cfg0, 10:3, secure_scratch47, 29:22);
	s(McVideoProtectVprOverride, 9:8, secure_scratch47, 31:30);
	s(McEmemAdrCfgBankMask2, 31:10, secure_scratch48, 21:0);
	s(McGeneralizedCarveout3Cfg0, 21:14, secure_scratch48, 29:22);
	s(McVideoProtectVprOverride, 11:11, secure_scratch48, 30:30);
	s(McVideoProtectVprOverride, 14:14, secure_scratch48, 31:31);
	s(McVideoProtectGpuOverride1, 15:0, secure_scratch49, 15:0);
	s(McEmemCfg, 13:0, secure_scratch49, 29:16);
	s(McEmemCfg, 31:31, secure_scratch49, 30:30);
	s(McVideoProtectVprOverride, 15:15, secure_scratch49, 31:31);
	s(McGeneralizedCarveout3Bom, 31:17, secure_scratch50, 14:0);
	s(McGeneralizedCarveout1Bom, 31:17, secure_scratch50, 29:15);
	s(McVideoProtectVprOverride, 18:17, secure_scratch50, 31:30);
	s(McGeneralizedCarveout4Bom, 31:17, secure_scratch51, 14:0);
	s(McGeneralizedCarveout2Bom, 31:17, secure_scratch51, 29:15);
	s(McVideoProtectVprOverride, 20:19, secure_scratch51, 31:30);
	s(McGeneralizedCarveout5Bom, 31:17, secure_scratch52, 14:0);
	s(McVideoProtectBom, 31:20, secure_scratch52, 26:15);
	s(McVideoProtectVprOverride, 23:21, secure_scratch52, 29:27);
	s(McVideoProtectVprOverride, 26:26, secure_scratch52, 30:30);
	s(McVideoProtectVprOverride, 29:29, secure_scratch52, 31:31);
	s(McVideoProtectSizeMb, 11:0, secure_scratch53, 11:0);
	s(McSecCarveoutBom, 31:20, secure_scratch53, 23:12);
	s(McVideoProtectVprOverride, 31:30, secure_scratch53, 25:24);
	s(McVideoProtectVprOverride1, 1:0, secure_scratch53, 27:26);
	s(McVideoProtectVprOverride1, 7:4, secure_scratch53, 31:28);
	s(McSecCarveoutSizeMb, 11:0, secure_scratch54, 11:0);
	s(McMtsCarveoutBom, 31:20, secure_scratch54, 23:12);
	s(McVideoProtectVprOverride1, 15:8, secure_scratch54, 31:24);
	s(McMtsCarveoutSizeMb, 11:0, secure_scratch55, 11:0);
	s(McGeneralizedCarveout4Size128kb, 11:0, secure_scratch55, 23:12);
	s(McVideoProtectVprOverride1, 16:16, secure_scratch55, 24:24);
	s(McGeneralizedCarveout2Cfg0, 2:0, secure_scratch55, 27:25);
	s(McGeneralizedCarveout2Cfg0, 25:22, secure_scratch55, 31:28);
	s(McGeneralizedCarveout3Size128kb, 11:0, secure_scratch56, 11:0);
	s(McGeneralizedCarveout2Size128kb, 11:0, secure_scratch56, 23:12);
	s(McGeneralizedCarveout2Cfg0, 26:26, secure_scratch56, 24:24);
	s(McGeneralizedCarveout1Cfg0, 2:0, secure_scratch56, 27:25);
	s(McGeneralizedCarveout1Cfg0, 25:22, secure_scratch56, 31:28);
	s(McGeneralizedCarveout1Size128kb, 11:0, secure_scratch57, 11:0);
	s(McGeneralizedCarveout5Size128kb, 11:0, secure_scratch57, 23:12);
	s(McGeneralizedCarveout1Cfg0, 26:26, secure_scratch57, 24:24);
	s(McGeneralizedCarveout3Cfg0, 2:0, secure_scratch57, 27:25);
	s(McGeneralizedCarveout3Cfg0, 25:22, secure_scratch57, 31:28);
	s(McGeneralizedCarveout3Cfg0, 26:26, secure_scratch58, 0:0);

	s32(McGeneralizedCarveout1Access0, secure_scratch59);
	s32(McGeneralizedCarveout1Access1, secure_scratch60);
	s32(McGeneralizedCarveout1Access2, secure_scratch61);
	s32(McGeneralizedCarveout1Access3, secure_scratch62);
	s32(McGeneralizedCarveout1Access4, secure_scratch63);
	s32(McGeneralizedCarveout2Access0, secure_scratch64);
	s32(McGeneralizedCarveout2Access1, secure_scratch65);
	s32(McGeneralizedCarveout2Access2, secure_scratch66);
	s32(McGeneralizedCarveout2Access3, secure_scratch67);
	s32(McGeneralizedCarveout2Access4, secure_scratch68);
	s32(McGeneralizedCarveout3Access0, secure_scratch69);
	s32(McGeneralizedCarveout3Access1, secure_scratch70);
	s32(McGeneralizedCarveout3Access2, secure_scratch71);
	s32(McGeneralizedCarveout3Access3, secure_scratch72);
	s32(McGeneralizedCarveout3Access4, secure_scratch73);
	s32(McGeneralizedCarveout4Access0, secure_scratch74);
	s32(McGeneralizedCarveout4Access1, secure_scratch75);
	s32(McGeneralizedCarveout4Access2, secure_scratch76);
	s32(McGeneralizedCarveout4Access3, secure_scratch77);
	s32(McGeneralizedCarveout4Access4, secure_scratch78);
	s32(McGeneralizedCarveout5Access0, secure_scratch79);
	s32(McGeneralizedCarveout5Access1, secure_scratch80);
	s32(McGeneralizedCarveout5Access2, secure_scratch81);
	s32(McGeneralizedCarveout5Access3, secure_scratch82);
	s32(McGeneralizedCarveout1ForceInternalAccess0, secure_scratch84);
	s32(McGeneralizedCarveout1ForceInternalAccess1, secure_scratch85);
	s32(McGeneralizedCarveout1ForceInternalAccess2, secure_scratch86);
	s32(McGeneralizedCarveout1ForceInternalAccess3, secure_scratch87);
	s32(McGeneralizedCarveout1ForceInternalAccess4, secure_scratch88);
	s32(McGeneralizedCarveout2ForceInternalAccess0, secure_scratch89);
	s32(McGeneralizedCarveout2ForceInternalAccess1, secure_scratch90);
	s32(McGeneralizedCarveout2ForceInternalAccess2, secure_scratch91);
	s32(McGeneralizedCarveout2ForceInternalAccess3, secure_scratch92);
	s32(McGeneralizedCarveout2ForceInternalAccess4, secure_scratch93);
	s32(McGeneralizedCarveout3ForceInternalAccess0, secure_scratch94);
	s32(McGeneralizedCarveout3ForceInternalAccess1, secure_scratch95);
	s32(McGeneralizedCarveout3ForceInternalAccess2, secure_scratch96);
	s32(McGeneralizedCarveout3ForceInternalAccess3, secure_scratch97);
	s32(McGeneralizedCarveout3ForceInternalAccess4, secure_scratch98);
	s32(McGeneralizedCarveout4ForceInternalAccess0, secure_scratch99);
	s32(McGeneralizedCarveout4ForceInternalAccess1, secure_scratch100);
	s32(McGeneralizedCarveout4ForceInternalAccess2, secure_scratch101);
	s32(McGeneralizedCarveout4ForceInternalAccess3, secure_scratch102);
	s32(McGeneralizedCarveout4ForceInternalAccess4, secure_scratch103);
	s32(McGeneralizedCarveout5ForceInternalAccess0, secure_scratch104);
	s32(McGeneralizedCarveout5ForceInternalAccess1, secure_scratch105);
	s32(McGeneralizedCarveout5ForceInternalAccess2, secure_scratch106);
	s32(McGeneralizedCarveout5ForceInternalAccess3, secure_scratch107);

	/* Locking PMC secure scratch register (8 ~ 15) for writing */
	c(0x5555, sec_disable2, 15:0);
	/* Locking PMC secure scratch register (4~ 7) for both reading and writing */
	c(0xff, sec_disable, 19:12);

	c32(0, scratch2);
	s(PllMInputDivider, 7:0, scratch2, 7:0);
	s(PllMFeedbackDivider, 7:0, scratch2, 15:8);
	s(PllMPostDivider, 4:0, scratch2, 20:16);
	s(PllMKVCO, 0:0, scratch2, 21:21);
	s(PllMKCP, 1:0, scratch2, 23:22);

	c32(0, scratch35);
	s(PllMSetupControl, 15:0, scratch35, 15:0);

	c32(0, scratch3);
	s(PllMInputDivider, 7:0, scratch3, 7:0);
	c(0x3e, scratch3, 15:8);
	c(0, scratch3, 20:16);
	s(PllMKVCO, 0:0, scratch3, 21:21);
	s(PllMKCP, 1:0, scratch3, 23:22);

	c32(0, scratch36);
	s(PllMSetupControl, 23:0, scratch36, 23:0);

	c32(0, scratch4);
	s(PllMStableTime, 9:0, scratch4, 9:0);
}
